using System;
using System.Collections;


namespace DarkStrideToolbox
{
	public class DSBitArray
	{
		#region Properties
		private int m_nMaxIndexAllocated = 0;
		private int m_nMaxIndexUsed = -1;
		private BitArray m_oBits = null;
		private object m_oSourceEntity = null;
		private bool[] m_baSourceEntityReadIn = null;
		#endregion


		public DSBitArray( int nLength )
		{
			m_nMaxIndexAllocated = DSMisc.Max( 10,nLength );
			m_oBits = new BitArray( m_nMaxIndexAllocated );			
		}

		public DSBitArray( byte[] oaBytes )
		{
			m_oBits = new BitArray( oaBytes );
			m_nMaxIndexAllocated = m_oBits.Length;
			m_nMaxIndexUsed = m_nMaxIndexAllocated - 1;
		}



		public void Set( int nIndex,bool bValue )
		{
			MakeSureIndexIsAvailable( nIndex );
			CheckSourceEntity( nIndex );

			m_oBits[ nIndex ] = bValue;
			m_nMaxIndexUsed = DSMisc.Max( m_nMaxIndexUsed,nIndex );
		}
		public bool Get( int nIndex )
		{
			CheckSourceEntity( nIndex );

			if( nIndex <= m_nMaxIndexUsed )
			{
				return( m_oBits[ nIndex ] );
			}
			else
			{
				return( false );
			}
		}

		public override string ToString()
		{
			string sRetVal = "";

			for( int i=0 ; i<=m_nMaxIndexUsed ; i++ )
			{
				if( Get( i ) == true )
				{
					sRetVal += "1";
				}
				else
				{
					sRetVal += "0";
				}
			}

			return( sRetVal );
		}


		private void MakeSureIndexIsAvailable( int nIndex )
		{
			if( nIndex >= m_nMaxIndexAllocated )
			{
				ReAllocate( DSMisc.Max( m_nMaxIndexAllocated * 2,nIndex ) );
			}
		}
		private void CheckSourceEntity( int nIndex )
		{
			int nEntityIndex = (int)Math.Floor( (double)nIndex / 8.0 );
			BitArray oTempBitArray = null;
			byte[] oaBytes = null;
			string sChars = "";


			//So were using a source entity... have we copied over the element they are looking for yet?
			if( m_oSourceEntity != null && m_baSourceEntityReadIn[ nEntityIndex ] == false )
			{
				if( m_oSourceEntity.GetType() == typeof( byte[] ) )
				{
					oaBytes = (byte[])m_oSourceEntity;
					oTempBitArray = new BitArray( new byte[]{ oaBytes[ nEntityIndex ] } );
				}
				else if( m_oSourceEntity.GetType() == typeof( string ) )
				{
					sChars = (string)m_oSourceEntity;
					/*char cChar = sChars[ nEntityIndex ];
					oTempBitArray = new BitArray( new byte[]{ (byte)(int)cChar } );*/
					oTempBitArray = DSMisc.ConvertStringToBitArray( sChars.Substring( nEntityIndex,1 ) );
				}

				MakeSureIndexIsAvailable( ( nEntityIndex + 1 ) * 8 );
				for( int i=0 ; i<8 ; i++ )
				{
					//Set( nEntityIndex * 8 + i,oTempBitArray[ i ] );
					m_oBits[ nEntityIndex * 8 + i ] = oTempBitArray[ i ];
				}				
				m_baSourceEntityReadIn[ nEntityIndex ] = true;
			}
		}

		private void ReAllocate( int nNewMax )
		{
			m_oBits = GetBitArray( nNewMax );
			m_nMaxIndexAllocated = nNewMax;
			if( m_oSourceEntity == null )
			{
				m_nMaxIndexUsed = DSMisc.Min( m_nMaxIndexUsed,nNewMax );
			}
		}

		private BitArray GetBitArray( int nNewMax )
		{
			BitArray oTempBits = new BitArray( nNewMax+1 );

			for( int i=0 ; i<=m_nMaxIndexUsed && i<nNewMax+1 ; i++ )
			{
				if( m_oSourceEntity != null && m_baSourceEntityReadIn[ (int)Math.Floor( (double)i / 8.0 ) ] == false )
				{
					i += 7;
				}
				else 
				{
					oTempBits[ i ] = Get( i );
				}
			}

			return( oTempBits );
		}
		public void Insert( int nIndexToInsertAt,int nNumToInsert )
		{
			BitArray oTempBits = GetBitArray( m_nMaxIndexAllocated + nNumToInsert );

			for( int i=m_nMaxIndexUsed ; i>=nIndexToInsertAt ; i-- )
			{
				oTempBits[ i + nNumToInsert ] = oTempBits[ i ];
				if( i >= nIndexToInsertAt && i < nIndexToInsertAt + nNumToInsert )
				{
					oTempBits[ i ] = false;
				}
			}

			m_oBits = oTempBits;
			m_nMaxIndexAllocated = m_nMaxIndexAllocated + nNumToInsert;
			if( nIndexToInsertAt < m_nMaxIndexUsed )
			{
				m_nMaxIndexUsed += nNumToInsert;
			}
		}


		public static DSBitArray ConvertStringToBitArray( string sStringToConvert )
		{
			DSBitArray oRetVal = new DSBitArray( 0 );
			BitArray oBitArrayValues = DSMisc.ConvertStringToBitArray( sStringToConvert );

			oRetVal.InternalBitArray = oBitArrayValues;
			oRetVal.Length = sStringToConvert.Length * 8;

			return( oRetVal );
		}
		public static string ConvertBitArrayToString( DSBitArray oValues )
		{
			string sRetVal = DSMisc.ConvertBitArrayToString( oValues.InternalBitArray );
			return( sRetVal );
		}
		public static void ConvertIntToBitArray( DSBitArray oSerialized,int nStartPos,long nNumBytesToUse,long nValue )
		{
			long nTemp = nValue;
			int nPos = nStartPos;


			//Calculate how many didgets needed
			for( int i=0 ; i<nNumBytesToUse ; i++ )
			{
				oSerialized.Set( i + nStartPos , ( ( nTemp & 1 ) == 1 ) );
				nTemp = nTemp >> 1;
			}
		}
		public static long ConvertBitArrayToInt( DSBitArray oSerialized,int nStartPos,long nNumBytesToUse )
		{
			long nTemp = 0;
			long nTotal = 0;


			//Calculate our value
			for( int nPos=0 ; nPos<nNumBytesToUse ; nPos++ )
			{
				nTemp = 0;
				if( nPos+nStartPos >= oSerialized.Length )
				{
					nTemp = 0;
				}
				else if( oSerialized.Get( nPos+nStartPos ) == true )
				{
					nTemp = 1;
				}
				nTemp = nTemp << nPos;
				nTotal += nTemp;
			}


			return( nTotal );
		}

		public static int GetNumberOfBitsUsedIn( int nValue )
		{
			return( DSBitArray.GetNumberOfBitsUsedIn( (long)nValue ) );
		}
		public static int GetNumberOfBitsUsedIn( double nValue )
		{
			return( DSBitArray.GetNumberOfBitsUsedIn( (long)nValue ) );
		}
		public static int GetNumberOfBitsUsedIn( long nValue )
		{
			long nTemp = nValue;
			int nDidgetsUsed = 0;

			while( nTemp > 0 )
			{
				nTemp = nTemp >> 1;
				nDidgetsUsed++;
			}

			return( nDidgetsUsed );
		}



		#region Properties
		public int Length
		{
			get
			{
				return( m_nMaxIndexUsed + 1 );
			}
			set
			{
				m_nMaxIndexUsed = value - 1;
				MakeSureIndexIsAvailable( m_nMaxIndexUsed );
			}
		}
		public BitArray InternalBitArray
		{
			get
			{
				BitArray oTempBits = GetBitArray( m_nMaxIndexUsed );
				return( oTempBits );
			}
			set
			{
				m_oBits = value;
                m_nMaxIndexUsed = m_oBits.Length - 1;
				m_nMaxIndexAllocated = m_oBits.Length - 1;
			}
		}
		public object SourceEntity
		{
			get
			{
				return( m_oSourceEntity );
			}
			set
			{
				if( value.GetType() == typeof( byte[] ) )
				{
					byte[] oaBytes = (byte[])value;
					m_oSourceEntity = oaBytes;
					m_baSourceEntityReadIn = new bool[ oaBytes.Length ];
					m_nMaxIndexUsed = oaBytes.Length * 8 - 1;
				}
				else if( value.GetType() == typeof( string ) )
				{
					string sValue = (string)value;
					m_oSourceEntity = sValue;
					m_baSourceEntityReadIn = new bool[ sValue.Length ];
					m_nMaxIndexUsed = sValue.Length * 8 - 1;
				}
				else
				{
					throw new System.Exception( "A source entity can only be of type string or byte array." );
				}
			}
		}
		#endregion
	}
}
