using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.IO;
using System.Threading;
using System.Runtime.InteropServices;

using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;
using Microsoft.DirectX.DirectPlay;
using Microsoft.DirectX.DirectInput;
using Microsoft.DirectX.DirectSound;

/*
ToDo For Game Engine
--------------------
Add a way for loaded in forms to be removed, but then retreived if you do a show on them
Add font capability to controls
Find out why there is blue at the edge of the horz scroll bar
Reinit mouse if its not inited properly the first time
System is not closing when an exception occurs
And make it so errors are meaningfull if graphics files are not found
Add way to trap enter hit in text box
Fix it so if you type over the length of a text box characters fall off the beginning
Get Ctrl clicks in list boxes to work
Make forms visible
Make forms have move event
Make forms have center on startup property
Make controls use standard names, events and properties
Strip out error handling and replace with new System  
Add a way to shut down the network after its started
Make non-exclusive mouse mode do better with the cursor, then hide our fake cursor
Figure out how to determine the length text will take to display
Make the label control truncte text that is too long
And change color of focus bar if the control isn't the focus
add property for what is Selected to list box control
Add click and key events to change selected item in list box
Add ability to deselect items in list box control
*/
namespace DarkStrideToolbox
{
	public class DSGameEngineArgs: EventArgs 
	{
		public DSGameEngineArgs( double dElapsedTime,double dAppTime ) 
		{
			this.m_dElapsedTime = dElapsedTime;
			this.m_dAppTime = dAppTime;
		}

		private double m_dElapsedTime;
		private double m_dAppTime;
		public double ElapsedTime
		{
			get
			{
				return( m_dElapsedTime );
			}
		}
		public double AppTime
		{
			get
			{
				return( m_dAppTime );
			}
		}
	}

	public class DSGameEngine : DSGraphicsWrapper
	{
		#region Properties
		#region FPS Functions & Timing API Calls
		private string m_sUserFrameStats = "";
		private double m_nSecondsSpentRendering = 0;
		private double m_nSecondsSpentMoving = 0;

		//This is the number of times we have rendered the screen in one second
		private double m_dFPS = 0.0;
		private long m_nFramesSinceLastFPSCalc = 0;
		private double m_dAppTime = 0.0;				// Current time in seconds
		private double m_dElapsedTime = 0.0;			// Time elapsed since last frame
		private double m_dElapsedTimeSinceLastFPSCalc = 0.0;	// Time elapsed since last frame
		private double m_dLastFrameTime = 0;		// The last time
		//private long m_nAppStartTime = 0;
		//private long m_nLastFPSCalcTime = 0;		// The last time			
		private long m_nLastCounterQuery = 0;		// The last time			
		private long m_nTicksPerSecond = 0;
		private long m_nPolygonsRenderedLastFPSCalc = 0;
		private long m_nPolygonsRenderedForCurrentFPSCalc = 0;
		private long m_nPolygonsRenderedSinceLastFrame = 0;
		private long m_nPolygonsRenderedPerFrame = 0;

		//07/15/2005 Chris Hill  This doesn't appear to be working so turning it off by default for now.
		//11/30/2004 Chris Hill  This is the max frame rate we will allow... roughly.  The best human eye can only handle 72 
		//frames per second, the average only 60.  Their is no sence in the CPU dying from overwork to display 350 FPS.  This
		//is also to help keep the game running smooth as graphics cause the FPS to dip and rise.  The second variable is to 
		//keep track of how much FPS we've accumulated so far.  Keep in mind that with DX timers calling them to get their
		//value also resets them (for get elapsedtime).
		private double m_nMaxFPS = 0.0f;
		#endregion

		#region Font Stuff
		// Interop to call get device caps
		private const int m_cLOGPIXELSY = 90;
		[System.Runtime.InteropServices.DllImport("gdi32.dll")]
		private static extern int GetDeviceCaps(IntPtr hdc, int cap);
		#endregion

		private DSSortedList m_oLoadedStateBlocks = new DSSortedList();

		private DSParticleSystem m_oParticleSystem = null;
		private const string m_cPARTICLESYSTEM_STATEBLOCKKEY = "Particle System State Block Key";
		private DSForms m_oForms = null;
		
		private bool m_bPaused = false;		//11/30/2004 Chris Hill  This changes the advancement of the system not the rendering.  It is very useful for pausing a game.
		private bool m_bDisposed = false;
	
		#region DirectSound Variables
		private DSSoundWrapper m_oSound = null;
		#endregion
		#region DirectInput Variables
		private DSGameEngineInput m_oInput = null;
		//08/17/2006 Chris Hill  Switched to a hardware rendered mouse so this is no longer necessary.
		//private bool m_bShowMouseCursor = true;
		private LoadedTexture m_oMouseCursor_Texture;
		private const string m_cMOUSECURSOR_VERTEXBUFFERKEY = "SYS_MouseCursor_VertexBuffer";
		private Vector2 m_cMOUSECURSORSIZE = new Vector2( 14,14 );
		public const string m_cMOUSECURSORNAME = "System_MouseCursor_";
		#endregion
		#region DirectPlay Variables
		private DSNetworkBuffered m_oNetwork = null;
		private System.Exception m_oDirectPlayError = null;
		#endregion
		
		private System.Windows.Forms.Control m_oForm = null;
		private Vector2 m_vFullScreenResolution = new Vector2( 0,0 );
		private bool m_bFullScreen = false;

		private System.Drawing.Color m_oBackgroundClearColor = System.Drawing.Color.Black;

        private DSBufferDebug m_oDebug = null;

		#region Event Handlers
		//public event EventHandler Dispose;		
		public event EventHandler OneTimeSceneInitialization;
		public event EventHandler SetDeviceSettings;
		public event EventHandler SetRenderStates;
		public event EventHandler FrameMove;
		public event EventHandler RenderBeforeForms;
		public event EventHandler RenderAfterForms;
		//public event EventHandler Render3D;
		public event EventHandler DeviceRestored;
		#endregion
		#endregion	


		public DSGameEngine( System.Windows.Forms.Control oForm )
		{
            Constructor(oForm, "", false);
		}
		public DSGameEngine( System.Windows.Forms.Control oForm,string sDebugPath,bool bDebugMode )
        {
            Constructor( oForm,sDebugPath,bDebugMode );
		}
		private void Constructor( System.Windows.Forms.Control oForm,string sDebugPath,bool bDebugMode )
		{
            m_oDebug = new DSBufferDebug();
            if (sDebugPath.Length > 0)
            {
                m_oDebug.DebugPath = sDebugPath;
            }
            else
            {
                m_oDebug.DebugPath = DSMisc.GetDevelopmentAppPath() +
                                        "DSGameEngine_DebugFile_" +
                                        DateTime.Now.ToString("yyyy-MM-dd hh") + "h" +
                                        DateTime.Now.ToString("mm") + "m" +
                                        DateTime.Now.ToString("ss") + "s" +
                                        ".OUT";
            }
            m_oDebug.DebugMode = bDebugMode;

			m_oForm = oForm;
		}
		private void InitializeDevices()
		{
			long nCurrentTime = 0;


			m_nTicksPerSecond = DSMisc.GetQueryPerformanceFrequency();
			nCurrentTime = DSMisc.GetQueryPerformanceCounter();

            m_oNetwork = new DSNetworkBuffered();//"", m_oDebug.DebugMode);
            m_oNetwork.DebugMode = m_oDebug.DebugMode;

			if( m_oForm != null )
			{
				//Initialize the graphics engine
				Initialize( m_oForm,m_bFullScreen,m_vFullScreenResolution );

				//Initialize our sound engine
				m_oSound = new DSSoundWrapper( m_oForm );
				//Initialize our global resource cache
				DSResourceManager.GetGlobalInstance().OnCreateDevice( Direct3DDevice );
				//Load our embeded resources
				LoadEmbededResourcesFromProject();

				//Initialize our input
                m_oInput = new DSGameEngineInput();//"", m_oDebug.DebugMode);
                m_oInput.DebugMode = m_oDebug.DebugMode;

				m_oInput.Initialize( this.Form );
				//Initialize our menu system
				m_oForms = new DSForms( m_oInput,this );
				m_oInput.Forms = m_oForms;
				//Initialize our particle system
				m_oParticleSystem = new DSParticleSystem( this.Direct3DDevice );
				//Take care of things like resizing so we can handle our changed device
				this.Form.Resize += new EventHandler( this.OnResize );

				//Initialize our mouse cursor message pump
				if( m_oForm.GetType().BaseType == typeof( DSWinForm ) )
				{
					((DSWinForm)m_oForm).MessageReceived += new EventHandler( this.OnWndProcMsg );
					SetupTheMouse();
				}
			}

			//Initialize our network
			m_oNetwork.RegisterDelegate_Error( new DSNetworkWrapper.CallbackError( this.NetworkError ) );
		}

		private void LoadEmbededResourcesFromProject()
		{
			System.Reflection.Assembly oAssembly = null;
			string[] saResourceNames = null;
			string sResKey = "";
			string sResFileName = "";
			string sSourceFolder = "DarkStrideToolbox.Graphics.";
			int nPos = 0;


			oAssembly = System.Reflection.Assembly.GetExecutingAssembly();
			saResourceNames = oAssembly.GetManifestResourceNames();

			//Walk down the list of resources in the graphics directory and load them in
			for( int i=0; i<saResourceNames.Length ; i++ )
			{
				sResFileName = saResourceNames[i];
				if( sResFileName.Length >= sSourceFolder.Length )
				{
					if( sResFileName.Substring( 0,sSourceFolder.Length ).Equals( sSourceFolder ) == true )
					{
						nPos = sResFileName.IndexOf( "Graphics.",0 ) + 1;
						sResKey = sResFileName.Substring( sSourceFolder.Length );
						sResKey = "System_" + sResKey.Substring( 0,sResKey.Length - 4 );

						if( DSMisc.Left( sResKey,m_cMOUSECURSORNAME.Length ).Equals( m_cMOUSECURSORNAME ) == true )
						{		
							DSResourceManager.GetGlobalInstance().LoadTexture( sResKey,sResFileName,System.Drawing.Color.Black,true );
						}
						else
						{
							DSResourceManager.GetGlobalInstance().LoadTexture( sResKey,sResFileName,true );
						}
					}
				}
			}
		}

		public void Dispose()
		{
			m_bDisposed = true;
		}
		public void Run()
		{
			double nTimeToWait  = 0;
			double nNewElapsedTime = 0;
			long nCurrentTime = 0;
			long nRenderStart = 0;
			long nRenderStop = 0;
			long nMoveStop = 0;


			RaiseSetDeviceSettingsEvent();
			InitializeDevices();
			RaiseSetRenderStatesEvent();
			RaiseOneTimeSceneInitializationEvent();
			
			if( this.Form != null )
			{
				this.TopmostForm.Show();
			}

			//04/01/2005 Chris Hill  Interesting, C# stops processing the line if the first paramater isn't valid.  Handy!
			while( m_bDisposed == false &&
					(
						this.Form == null || ( this.TopmostForm != null && this.TopmostForm.Created == true ) 
				    )
				 )
			{
				m_oDebug.WriteToDebug( 0,"Start Frame Advancement..." );

				//Calculate our time changes
				m_nLastCounterQuery = nCurrentTime;
				nCurrentTime = DSMisc.GetQueryPerformanceCounter();

				//Don't let the first time check be huge
				if( m_nLastCounterQuery == 0 )
				{
					m_nLastCounterQuery = nCurrentTime;
				}
				nNewElapsedTime = ( (double)( nCurrentTime - m_nLastCounterQuery ) ) / (double)m_nTicksPerSecond;
				m_dElapsedTime += nNewElapsedTime;
				m_dElapsedTimeSinceLastFPSCalc += nNewElapsedTime;

				//We don't add in the first one so that app time starts at 0.
				if( m_nLastCounterQuery != 0 )
				{
					m_dAppTime += nNewElapsedTime;
				}

				//12/06/2006 Chris Hill  Are we over-rendering?  The human eye can only see around 40-60 FPS.
				//So many times we want to slow down the rendering, why waste the CPU.  
				if( m_dElapsedTime < 1.0f / m_nMaxFPS && m_nMaxFPS != 0 )
				{
					m_dElapsedTime = 0;
					m_dElapsedTimeSinceLastFPSCalc -= nNewElapsedTime;
					m_dAppTime -= nNewElapsedTime;


					//Start by calculating how long we need to wait to hit exactly our FPS rating, then wait
					nTimeToWait = ( ( 1.0f / m_nMaxFPS ) - m_dElapsedTime ) * 1000;
					m_oDebug.WriteToDebug( 1,"Too Fast To Render, ElapsedTime:" + m_dElapsedTime.ToString() + " Sleep:" + nTimeToWait.ToString() );
					System.Threading.Thread.Sleep( (int)nTimeToWait );

					nCurrentTime = DSMisc.GetQueryPerformanceCounter();

					nNewElapsedTime = ( (double)( nCurrentTime - m_nLastCounterQuery ) ) / (double)m_nTicksPerSecond;
					m_oDebug.WriteToDebug( 1,"...Finished Sleeping, Time passed:" + nNewElapsedTime.ToString() );
					m_dElapsedTime += nNewElapsedTime;
					m_dElapsedTimeSinceLastFPSCalc += nNewElapsedTime;
					m_dAppTime += nNewElapsedTime;
				}

				m_dLastFrameTime = m_dAppTime;
				m_nPolygonsRenderedPerFrame = this.PolygonsRendered - m_nPolygonsRenderedSinceLastFrame;
				m_nPolygonsRenderedSinceLastFrame = this.PolygonsRendered;

				if( m_dElapsedTimeSinceLastFPSCalc >= 1.0f )
				{
					m_nPolygonsRenderedForCurrentFPSCalc = this.PolygonsRendered - m_nPolygonsRenderedLastFPSCalc;
					m_nPolygonsRenderedLastFPSCalc = this.PolygonsRendered;
					m_dFPS = (double)m_nFramesSinceLastFPSCalc / m_dElapsedTimeSinceLastFPSCalc;
					m_dElapsedTimeSinceLastFPSCalc = 0.0;
					m_nFramesSinceLastFPSCalc = 0;
				}

				CheckForNetworkErrors();

				//Track some timing information
				nRenderStart = DSMisc.GetQueryPerformanceCounter();

				//If we are minimized don't render
				if( this.Form != null && this.TopmostForm.WindowState != System.Windows.Forms.FormWindowState.Minimized )
				{
					//Render the scene
					OnRender();
				}
				//State update
				m_nFramesSinceLastFPSCalc++;

				//Track some timing information
				nRenderStop = DSMisc.GetQueryPerformanceCounter();

				//05/05/2007 Chris Hill  Changed the order of the frame moving so that things the
				//user does can affect the frame move of the engine.
				RaiseFrameMoveEvent( m_dElapsedTime,m_dAppTime );
				OnFrameMove( m_dElapsedTime,m_dAppTime );

				//Track some timing information
				nMoveStop = DSMisc.GetQueryPerformanceCounter();

				m_dElapsedTime = 0;


				//Yield some CPU time to other processes
				//If we aren't minimized still render but do so more slowly.
				if( this.Form != null && this.TopmostForm.WindowState == System.Windows.Forms.FormWindowState.Minimized )
				{
					System.Threading.Thread.Sleep( 100 );
				}
				else if( this.Form != null && ( this.HasFocus == false || System.Windows.Forms.Form.ActiveForm != this.TopmostForm ) )
				{
					System.Threading.Thread.Sleep( 10 );
				}
				System.Windows.Forms.Application.DoEvents();


				//Calculate our timing stats
				m_nSecondsSpentRendering = ( ( (double)nRenderStop - (double)nRenderStart ) / (double)m_nTicksPerSecond );
				m_nSecondsSpentMoving = ( ( (double)nMoveStop - (double)nRenderStop ) / (double)m_nTicksPerSecond );

				//Write out timing stats to the file
				m_oDebug.WriteToDebug( 1,this.FrameStats.ToString() );					
			}

			if( m_oInput != null )
			{
				m_oInput.Dispose();
			}
			if( m_oNetwork != null )
			{
				m_oNetwork.Dispose();
			}
			//RaiseDisposeEvent();
		}

		private void OnRender()
		{
			//11/20/2004 Chris Hill  Added debugging.
			m_oDebug.WriteToDebug( 2,"Starting Render3DEnvironment( Time:" + m_dAppTime.ToString() + " Elapsed:" + m_dElapsedTime.ToString() + " )" );

			int result = 0;
			if( this.Direct3DDevice != null && 
				//this.TopmostForm.WindowState != FormWindowState.Minimized &&
				//this.Form.Visible == true &&
				this.Direct3DDevice.CheckCooperativeLevel( out result ) )
			{  
				//Okay to render
				try
				{
					//04/21/2007 Chris Hill  I turned this off because in switching to fullscreen
					//it was messing things up when alt-tabbing out.  
					/*//Render the scene as normal
					if( this.Direct3DDevice.Viewport.X != 0 || this.Direct3DDevice.Viewport.Y != 0 ||
						this.Direct3DDevice.Viewport.Width != this.Form.ClientSize.Width || 
						this.Direct3DDevice.Viewport.Height != this.Form.ClientSize.Height )
					{
						//We are creating a viewport so we can be sure that our mouse cursor is on top
						oViewPort = this.Direct3DDevice.Viewport;
						oViewPort.X = 0;
						oViewPort.Y = 0;
						oViewPort.MinZ = 0;
						oViewPort.MaxZ = 0;
						oViewPort.Width = this.Form.ClientSize.Width;
						oViewPort.Height = this.Form.ClientSize.Height;
						this.Direct3DDevice.Viewport = oViewPort;
					}*/

					this.Direct3DDevice.Clear(ClearFlags.Target | ClearFlags.ZBuffer, m_oBackgroundClearColor, 1.0f, 0);
					this.Direct3DDevice.BeginScene();	


					///////////////////////////////////////////////////////////////////////////////////////////
					//Render Particle System
					///////////////////////////////////////////////////////////////////////////////////////////
					//CaptureStateBlock( m_cPARTICLESYSTEM_STATEBLOCKKEY );
					m_oParticleSystem.Render( this.WorldViewUpperLeft );
					//ApplyStateBlock( m_cPARTICLESYSTEM_STATEBLOCKKEY );
					///////////////////////////////////////////////////////////////////////////////////////////
					

					///////////////////////////////////////////////////////////////////////////////////////////
					//Start by rendering the 2D portion of the world
					///////////////////////////////////////////////////////////////////////////////////////////
					this.SpriteEngine.Begin( Microsoft.DirectX.Direct3D.SpriteFlags.AlphaBlend );				

					//this.Direct3DDevice.RenderState.Lighting = false;
					//this.Direct3DDevice.SetTransform( TransformType.View,Matrix.Identity );
					//this.Direct3DDevice.SetTransform( TransformType.Projection,Matrix.OrthoOffCenterLH( 0,this.ScreenWidth,this.ScreenHeight,0,0,1000 ) );

					RaiseRenderBeforeFormsEvent( m_dElapsedTime,m_dAppTime );
						
					//07/29/2005 Chris Hill  Render the forms after the user draws things, that way they will 
					//be on top of any backgrounds he wants.
					m_oForms.Render();

					//08/09/2005 Chris Hill  There are times we want to draw something on top of the forms, in
					//order to acomplish this we need a second event.
					RaiseRenderAfterFormsEvent( m_dElapsedTime,m_dAppTime );

					//08/16/2006 Chris Hill  I am now using a hardware rendered mouse, so this isn't needed.
					//Render the mouse cursor
					//RenderTheMouse();


					this.SpriteEngine.End();
					///////////////////////////////////////////////////////////////////////////////////////////

					//11/20/2004 Chris Hill  Added debugging.
					m_oDebug.WriteToDebug( 2,"Starting Direct3DDevice.Present" );
					this.Direct3DDevice.EndScene();
					this.Direct3DDevice.Present();
				}
				catch( Microsoft.DirectX.Direct3D.DeviceLostException )
				{
					this.Direct3DDevice.CheckCooperativeLevel(out result);
				}
				catch( Microsoft.DirectX.Direct3D.DeviceNotResetException )
				{
					this.Direct3DDevice.CheckCooperativeLevel(out result);
				}
			}

			if( result == (int)Microsoft.DirectX.Direct3D.ResultCode.DeviceLost )
			{
				Thread.Sleep(500);    //Can't Reset yet, wait for a bit
			}
			else if( result == (int)Microsoft.DirectX.Direct3D.ResultCode.DeviceNotReset )
			{
				this.Direct3DDevice.Reset( this.Direct3DDevice.PresentationParameters );
			}

			m_oDebug.WriteToDebug( 2,"...Starting Direct3DDevice.Present" );
		}
		private void OnResize( object sender, EventArgs e )
        {
			StateBlock oLoopStateBlock = null;


			//Go through all our state blocks and blow them away
			for( int i=0 ; i<m_oLoadedStateBlocks.Count ; i++ )
			{
				oLoopStateBlock = (StateBlock)m_oLoadedStateBlocks.GetByIndex( i );
				oLoopStateBlock.Dispose();
				oLoopStateBlock = null;
			}
			m_oLoadedStateBlocks = new DSSortedList();

			if( this.FullScreen == false )
			{
				this.Direct3DDevice.Reset( this.Direct3DDevice.PresentationParameters );
				DSResourceManager.GetGlobalInstance().OnRestore3DDeviceObjects();

				RaiseDeviceRestoredEvent();			
				RaiseSetRenderStatesEvent();
			}
		}
		private void SetupTheMouse()
		{
			string sTemporaryMouseCursorTextureKey = "";
			string sFinalMouseTextureKey = "";
			Microsoft.DirectX.Direct3D.Surface oSurface = null;
			//Viewport oViewPort;
			//System.Drawing.Rectangle oCursorLocation;
			LoadedTexture oTexture = null;


			//08/17/2006 Chris Hill  Switched to a hardware rendered mouse so this is no longer necessary.
			/*if( m_bShowMouseCursor == true )
			{
				//Do we have a standard screen sized viewport?  if not fix that.  
				if( this.Direct3DDevice.Viewport.X != 0 || this.Direct3DDevice.Viewport.Y != 0 ||
					this.Direct3DDevice.Viewport.Width != this.Form.ClientSize.Width || 
					this.Direct3DDevice.Viewport.Height != this.Form.ClientSize.Height )
				{
					//We are creating a viewport so we can be sure that our mouse cursor is on top
					oViewPort = this.Direct3DDevice.Viewport;
					oViewPort.X = 0;
					oViewPort.Y = 0;
					oViewPort.MinZ = 0;
					oViewPort.MaxZ = 0;
					oViewPort.Width = this.Form.ClientSize.Width;
					oViewPort.Height = this.Form.ClientSize.Height;
					this.Direct3DDevice.Viewport = oViewPort;
				}*/

				//Find out if the DSForm system wants to change the mouse cursor
				sTemporaryMouseCursorTextureKey = m_oForms.GetCurrentCursorTextureKey( this.DirectInput.MouseCursor );
				if( sTemporaryMouseCursorTextureKey != null && sTemporaryMouseCursorTextureKey.Length == 0 )
				{
					sTemporaryMouseCursorTextureKey = null;
				}

				//Do we have a temporary mouse cursor?  If so use that
				if( sTemporaryMouseCursorTextureKey != null )
				{
					sFinalMouseTextureKey = sTemporaryMouseCursorTextureKey;
				}
				//Do we have a preset texture?
				else if( m_oMouseCursor_Texture != null )
				{
					sFinalMouseTextureKey = m_oMouseCursor_Texture.sFileName;
				}
				//Otherwise use the built in mouse cursor
				else
				{
					sFinalMouseTextureKey = "";//System_MouseCursor_Hand";
				}

            	//Finally draw our mouse
				if( sFinalMouseTextureKey.Length > 0 )
				{
					oTexture = DSResourceManager.GetGlobalInstance().GetLoadedTexture( sFinalMouseTextureKey );
					/*oCursorLocation = new System.Drawing.Rectangle( (int)MouseCursor.X, (int)MouseCursor.Y,
																	(int)oTexture.Size.X, (int)oTexture.Size.Y );
					RenderTexture2D( sFinalMouseTextureKey, System.Drawing.Rectangle.Empty, oCursorLocation, 
									new Vector2(), 0, 0, false, System.Drawing.Color.White.ToArgb() );*/

					oSurface = Surface.FromStream( this.Direct3DDevice,oTexture.oStream,Pool.Default );
					//oSurface = Surface.FromBitmap( this.Direct3DDevice, bm, Pool.Default ); 
					this.Direct3DDevice.SetCursorProperties( 0,0,oSurface ); 
					this.Direct3DDevice.ShowCursor( true ); 

					oSurface.ReleaseGraphics();
					oSurface.Dispose();
				}
			//}
		}
		public void OnWndProcMsg( object sender, EventArgs e )
		{
			DSMessageArgs oArgs = (DSMessageArgs)e;


			if( oArgs.Msg == DSNativeMethods.WindowMessage.MouseMove ||
				oArgs.Msg == DSNativeMethods.WindowMessage.SetCursor )
			{
				this.Direct3DDevice.ShowCursor( true );
			}
		}
		private void OnFrameMove( double dElapsedTime,double dAppTime )
		{
			//11/20/2004 Chris Hill  Added debugging.
			m_oDebug.WriteToDebug( 1,"Starting FrameMove( Time:" + dAppTime.ToString() + " Elapsed:" + dElapsedTime.ToString() + " )" );

			//11/20/2004 Chris Hill  Added debugging.
			if( m_oInput != null )
			{
				m_oDebug.WriteToDebug( 2,"Starting m_oInput.ProccessBufferedEvents" );
				m_oInput.ProccessBufferedEvents();
			}
			//05/24/2006 Chris Hill  Get the network events.
			m_oNetwork.ProccessBufferedEvents();

			//11/30/2004 Chris Hill  If the scene is paused we don't advance but we do render.
			//This is neccessary because the buffered key events may have changed our status
			if( /*this.HasFocus == true &&*/ this.Paused == false && m_oParticleSystem != null )
			{
				//11/20/2004 Chris Hill  Added debugging.
				m_oDebug.WriteToDebug( 2,"Starting m_oParticleSystem.Advance" );

				m_oParticleSystem.Advance( dElapsedTime );
			}

			//11/20/2004 Chris Hill  Added debugging.
			m_oDebug.WriteToDebug( 2,"Starting m_odDelegate_FrameMove" );

			if( m_oForms != null )
			{
				m_oForms.FrameMove( dElapsedTime,dAppTime );
			}

			//11/20/2004 Chris Hill  Added debugging.
			m_oDebug.WriteToDebug( 1,"...Stopping FrameMove" );
		}

		#region State Block Functions
		public StateBlock CaptureStateBlock( string sKey )
		{
			StateBlock oStateBlock = null;

			
			oStateBlock = (StateBlock)m_oLoadedStateBlocks.GetByKey( sKey );
			if( oStateBlock == null )
			{
				oStateBlock = new StateBlock( this.Direct3DDevice,Microsoft.DirectX.Direct3D.StateBlockType.All );
				m_oLoadedStateBlocks.Add( sKey,oStateBlock );
			}
			oStateBlock.Capture(); 


			return( oStateBlock );
		}
		public void ApplyStateBlock( string sKey )
		{
			StateBlock oStateBlock = null;

			
			oStateBlock = (StateBlock)m_oLoadedStateBlocks.GetByKey( sKey );
			if( oStateBlock == null )
			{
				throw new System.Exception( "StateBlock with key of <" + sKey + "> not found." );
			}
			oStateBlock.Apply(); 
		}
		#endregion
		#region DirectPlay Functions
		private void NetworkError( System.Exception oError )
		{
			m_oDirectPlayError = DSMisc.MergeErrors( oError,m_oDirectPlayError );
		}

		private void CheckForNetworkErrors()
		{
			System.Exception oErrors = null;


			//Throw any errors that may have occured.
			if( m_oDirectPlayError != null )
			{
				lock( m_oDirectPlayError )
				{
					oErrors = m_oDirectPlayError;
					m_oDirectPlayError = null;
				}
				if( oErrors != null )
				{
					throw new System.Exception( "DirectPlay Threw an error",oErrors );
				}
			}
		}
		#endregion
		#region Font Functions
		public int GetFontSize( int nFontSize )
		{
			int nHeight = 0;
			int nLogPixelsY = 0;
			System.IntPtr nDC = System.IntPtr.Zero; 
			System.Drawing.Graphics oGraphics = null;


			if( this.Form.IsDisposed == false )
			{
				//Initialize all of the fonts
				oGraphics = this.Form.CreateGraphics();
				nDC = oGraphics.GetHdc();

				nLogPixelsY = GetDeviceCaps( nDC,m_cLOGPIXELSY );

				oGraphics.ReleaseHdc( nDC );

				nHeight = -nFontSize * nLogPixelsY / 72;
			}


			return( nHeight );
		}
		#endregion
		#region Event Handler Functions
		protected void RaiseOneTimeSceneInitializationEvent()
		{
			EventArgs oArgs = new EventArgs();

			if( OneTimeSceneInitialization != null )
			{
				OneTimeSceneInitialization( this,oArgs );
			}
		}
		protected void RaiseSetRenderStatesEvent()
		{
			EventArgs oArgs = new EventArgs();

			if( SetRenderStates != null )
			{
				SetRenderStates( this,oArgs );
			}
		}
		protected void RaiseSetDeviceSettingsEvent()
		{
			EventArgs oArgs = new EventArgs();

			if( SetDeviceSettings != null )
			{
				SetDeviceSettings( this,oArgs );
			}
		}		
		protected void RaiseFrameMoveEvent( double dElapsedTime,double dAppTime )
		{
			DSGameEngineArgs oArgs = null;

			if( FrameMove != null )
			{
				oArgs = new DSGameEngineArgs( dElapsedTime,dAppTime );
				FrameMove( this, oArgs );
			}
		}

		protected void RaiseRenderBeforeFormsEvent( double dElapsedTime,double dAppTime )
		{
			DSGameEngineArgs oArgs = null;

			if( RenderBeforeForms != null )
			{
				oArgs = new DSGameEngineArgs( dElapsedTime,dAppTime );
				RenderBeforeForms( this, oArgs );
			}
		}

		protected void RaiseRenderAfterFormsEvent( double dElapsedTime,double dAppTime )
		{
			DSGameEngineArgs oArgs = null;

			if( RenderAfterForms != null )
			{
				oArgs = new DSGameEngineArgs( dElapsedTime,dAppTime );
				RenderAfterForms( this, oArgs );
			}
		}

		/*protected void RaiseRender3DEvent( double dElapsedTime,double dAppTime )
		{
			DSGameEngineArgs oArgs = null;

			if( Render3D != null )
			{
				oArgs = new DSGameEngineArgs( dElapsedTime,dAppTime );
				Render3D( this, oArgs );
			}
		}*/
		protected void RaiseDeviceRestoredEvent()
		{
			EventArgs oArgs = null;

			if( DeviceRestored != null )
			{
				oArgs = new EventArgs();
				DeviceRestored( this, oArgs );
			}
		}
		

		/*protected void RaiseDisposeEvent()
		{
			EventArgs oArgs = new EventArgs();

			if( Dispose != null )
			{
				Dispose( this,oArgs );
			}
		}
*/
		#endregion

        public void WriteToDebug(int nLevel,string sMessage )
        {
            m_oDebug.WriteToDebug(nLevel, sMessage);
        }


		#region Properties
		public DSParticleSystem ParticleSystem
		{
			get
			{
				return( m_oParticleSystem );
			}
		}

		public bool MouseInExclusiveMode
		{
			get
			{
				return( m_oInput.MouseInExclusiveMode );
			}
			set
			{
				m_oInput.MouseInExclusiveMode = value;
				m_oInput.UseAbsoluteMouseCoordinates = !m_oInput.MouseInExclusiveMode;
			}
		}
		public DSForms Forms
		{
			get
			{
				return( m_oForms );
			}
		}
		//08/17/2006 Chris Hill  Switched to a hardware rendered mouse so this is no longer necessary.
		/*public bool ShowMouseCursor
		{
			get
			{
				return( m_bShowMouseCursor );
			}
			set
			{
				m_bShowMouseCursor = value;
			}
		}*/
		public Vector2 MouseCursor
		{
			get
			{
				return( m_oInput.MouseCursor );
			}
			set
			{
				m_oInput.MouseCursor = value;
			}
		}
		public bool[] MouseButtonState
		{
			get
			{
				return( m_oInput.MouseButtonState );
			}
		}
		public DSGameEngineInput DirectInput
		{
			get
			{
				return( m_oInput );
			}
		}
		public DSNetworkBuffered DirectPlay
		{
			get
			{
				return( m_oNetwork );
			}
			set
			{
				m_oNetwork = value;
			}
		}
		public System.Drawing.Color BackgroundClearColor
		{
			set
			{
				m_oBackgroundClearColor = value;
			}
			get
			{
				return( m_oBackgroundClearColor );
			}
		}
		//11/20/2004 Chris Hill  Added a debug mode for tracking a time problem.
        public bool DebugMode
        {
            set
            {
                //Are we turning it on when it was off?
                if (m_oDebug.DebugMode == false && value == true)
                {
                    //Then write a header
                    DSBufferDebug.Inst().WriteToDebug(value, 0,
                        "\r\n\r\nStarting Debug Log File For DSGameEngine\r\n" +
                        "--------------------------------------------");
                }

                m_oDebug.DebugMode = value;
            }
            get
            {
                return (m_oDebug.DebugMode);
            }
        }
        public string DebugPath
        {
            get
            {
                return (m_oDebug.DebugPath);
            }
            set
            {
                m_oDebug.DebugPath = value;

                //We turn off then on the debug mode so that it writes a new header to the log
                if (m_oDebug.DebugMode == true)
                {
                    this.DebugMode = false;
                    this.DebugMode = true;
                }
            }
        }
		//11/30/2004 Chris Hill  This is the max frame rate we will allow... roughly.  The best human eye can only handle 72 
		//frames per second, the average only 60.  Their is no sence in the CPU dying from overwork to display 350 FPS.  This
		//is also to help keep the game running smooth as graphics cause the FPS to dip and rise.
		public double MaxFPS
		{
			get
			{
				return( m_nMaxFPS );
			}
			set
			{
				m_nMaxFPS = value;
			}
		}
		public string MouseTextureKey
		{
			get
			{
				return( m_oMouseCursor_Texture.sFileName );
			}
			set
			{
				m_oMouseCursor_Texture = DSResourceManager.GetGlobalInstance().GetLoadedTexture( value );
				SetupTheMouse();
			}
		}
		public bool Paused
		{
			get
			{
				return( m_bPaused );
			}
			set
			{
				m_bPaused = value;
			}
		}


		public string FrameStats
		{
			get
			{
                double nLatency = m_oNetwork.Latency * 1000;

				string sFrameStats = 
					"FPS: " + m_dFPS.ToString("f2") + 
					"\nResolution: " + this.BackBufferSize + 
					"\nPolygons per Frame: " + m_nPolygonsRenderedPerFrame.ToString() +
					"\nRender Time: " + m_nSecondsSpentRendering.ToString("f4") + "s" +
					"\nMove Time: " + m_nSecondsSpentMoving.ToString("f4") + "s" +
                    "\nLatency: " + nLatency.ToString("0") + "ms" +
					m_sUserFrameStats;

				return( sFrameStats );
			}
		}
		public string UserFrameStats
		{
			get
			{
				return( m_sUserFrameStats );
			}
			set
			{
				m_sUserFrameStats = value;
			}
		}
		public double AppTime
		{
			get
			{
				return( m_dAppTime );
			}
		}
		public double ElapsedTime
		{
			get
			{
				return( m_dElapsedTime );
			}
		}
		public long PolygonsRenderedPerFrame
		{
			get
			{
				return( m_nPolygonsRenderedPerFrame );
			}
		}
		public long PolygonsRenderedPerSecond
		{
			get
			{
				return( m_nPolygonsRenderedForCurrentFPSCalc );
			}
		}

		public DSSoundWrapper DirectSound
		{
			get
			{
				return( m_oSound );
			}
		}
		public Vector2 FullScreenResolution
		{
			get
			{
				return( m_vFullScreenResolution );
			}
			set
			{
				m_vFullScreenResolution = value;
			}
		}
		public bool FullScreen
		{
			get
			{
				return( m_bFullScreen );
			}
			set
			{
				m_bFullScreen = value;
			}
		}
		#endregion
	}


	public class DSGameEngineInput : DSInputBuffered
	{
		#region Properties
		private DSForms m_oForms = null;
		#endregion

		public DSGameEngineInput(){}
        public DSGameEngineInput(string sDebugPath, bool bDebugMode)
        {
            this.DebugPath = sDebugPath;
            this.DebugMode = bDebugMode;
        }
		public override void ProccessBufferedEvents()
		{
			BufferedInputEvent oEvent = null;

			try
			{
				if( m_oForms == null )
				{
					throw new System.Exception( "GameEngineInput does not have a 'Forms' instance to process for." );
				}

				while( base.Events.Count > 0 )
				{
					oEvent = (BufferedInputEvent)base.Events[ 0 ];
					base.Events.RemoveAt( 0 );

					if( oEvent.nEvent == BufferedInputEventTypes.enumMouseWheelMove )
					{
						if( m_oForms.OnMouseWheelMove( (long)oEvent.oVariables[ 0 ] ) == false )
						{
							base.ExecuteMouseWheelMove( (long)oEvent.oVariables[ 0 ] );
						}
					}
					else if( oEvent.nEvent == BufferedInputEventTypes.enumMouseMoved )
					{
						if( m_oForms.OnMouseMove( (Vector2)oEvent.oVariables[ 0 ],(MouseState)oEvent.oVariables[ 1 ],(bool[])oEvent.oVariables[ 2 ] ) == false )
						{
							base.ExecuteMouseMoved( (Vector2)oEvent.oVariables[ 0 ],(MouseState)oEvent.oVariables[ 1 ],(bool[])oEvent.oVariables[ 2 ] );
						}
					}
					else if( oEvent.nEvent == BufferedInputEventTypes.enumMouseUp )
					{
						if( m_oForms.OnMouseUp( (Vector2)oEvent.oVariables[ 0 ],(long)oEvent.oVariables[ 1 ] ) == false )
						{
							base.ExecuteMouseUp( (Vector2)oEvent.oVariables[ 0 ],(long)oEvent.oVariables[ 1 ] );
						}
					}
					else if( oEvent.nEvent == BufferedInputEventTypes.enumMouseDown )
					{
						if( m_oForms.OnMouseDown( (Vector2)oEvent.oVariables[ 0 ],(long)oEvent.oVariables[ 1 ] ) == false )
						{
							base.ExecuteMouseDown( (Vector2)oEvent.oVariables[ 0 ],(long)oEvent.oVariables[ 1 ] );
						}
					}
					else if( oEvent.nEvent == BufferedInputEventTypes.enumKeyboardKeyDown )
					{
						if( m_oForms.OnKeyDown( (Key)oEvent.oVariables[ 0 ],(bool[])oEvent.oVariables[ 1 ] ) == false )
						{
							base.ExecuteKeyboardKeyDown( (Key)oEvent.oVariables[ 0 ],(bool[])oEvent.oVariables[ 1 ] );
						}
					}
					else if( oEvent.nEvent == BufferedInputEventTypes.enumKeyboardKeyUp )
					{
						if( m_oForms.OnKeyUp( (Key)oEvent.oVariables[ 0 ],(bool[])oEvent.oVariables[ 1 ] ) == false )
						{
							base.ExecuteKeyboardKeyUp( (Key)oEvent.oVariables[ 0 ],(bool[])oEvent.oVariables[ 1 ] );
						}
					}
				}
			}
			catch( System.Exception oEx )
			{
				DSMisc.ShowErrors( oEx );
			}
		}

		#region Properties
		public DSForms Forms
		{
			get
			{
				return( m_oForms );
			}
			set
			{
				m_oForms = value;
			}
		}
		#endregion
	}
}