using System;
using System.Windows.Forms;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using System.Timers;
using System.Collections;

using Microsoft.DirectX.DirectPlay;

namespace DarkStrideToolbox
{
	public struct FindHostsResponseInformation
	{
		public int LastFoundTime;
		public ApplicationDescription ApplicationDesc;
		public int RoundTripLatencyMs;
		public Address sender;
		public Address device;
		public override string ToString()
		{
			if (ApplicationDesc.MaxPlayers > 0)
				return ApplicationDesc.SessionName + " (" + ApplicationDesc.CurrentPlayers + "/" + ApplicationDesc.MaxPlayers + ") (" + RoundTripLatencyMs + "ms)";
			else
				return ApplicationDesc.SessionName + " (" + ApplicationDesc.CurrentPlayers + ") (" + RoundTripLatencyMs + "ms)";
		}
	};
	public enum enumSystemMessageTypes
	{
		PingPacketRequest,
		PingPacketReply,
		UDPPlayerJoinRequst,
		UDPPlayerJoinReply,
		UDPPlayerHasJoined,
		UDPPlayerHasQuit,
		UDP_ACK,
		UDPPlayerQuit
	}

	public class DSNetworkWrapper
	{
		#region Constants
		private const double m_cSECONDS_SINCE_ACK_WITHNO_FURTHER_REQUESTS = 40;
		#endregion

		#region Properties
        private System.Collections.Generic.List<long> m_oPacketsToIgnoreIfObsolete = new System.Collections.Generic.List<long>();

        private string m_sTempMyPlayerName = "";

		private int m_nDefaultPort = 2502;
		private int m_nLivelenessTimoutTimeInSeconds = 60;

		private DSSortedList m_oJoiningGUIDToSocketID = new DSSortedList();
		private DSNetworkPlayer m_oUs = null;
		private DSNetworkPlayer m_oPlayerWhoIsHost = null;

		//Pinging variables
		private double m_nSecondsBetweenPings = 3;
		private Thread m_oPingThread = null;

		private System.Net.Sockets.ProtocolType m_oProtocolType = System.Net.Sockets.ProtocolType.Unknown;
		private Guid m_oAppGUID = Guid.Empty;
		private bool m_bConnected = false;
		private bool m_bIAmTheHost = false;
		private bool m_bInitialized = false;

		private DSSortedList m_oPlayerList = new DSSortedList();

        private DSBufferDebug m_oDebug = null;

		//When you say you want to join a game, we need to have some way to identify that the 
		//"yea, you can join" message from the server is truely for us.  This GUID is how we do that.
		private string m_sMyGUIDForJoining = "";

		#region TCPIP Specific Paramaters
		private Address m_oOurAddress = null;
		private Address m_oSearchingHostAddress = null;
		private ApplicationDescription m_oAppDesc;
		private int m_nFindHostHandle = 0;
		private int m_nConnectHandle = 0;
		private bool m_bSearchingForGames = false;

		private System.Timers.Timer m_oTCPIPConnectingTimer = null;
		private Peer m_oPeerObject = null;
		#endregion

		#region UDP Specific Paramaters
		private static int m_cMAX_UDPMESSAGESIZE = 65500;//The maximum possible for UDP
		private byte[] m_oaUDPMessageReceiveBuffer = new byte[ m_cMAX_UDPMESSAGESIZE ]; 

		private Socket m_oUDPServer = null;

		private Socket m_oUDPClient = null;//new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);
		//private UdpClient m_oUDPClient = null;

		private IPEndPoint m_oIEndPoint = null;
		private Thread m_oBlockedUDPDataReceiving = null;
		private Thread m_oUDP_ACKCheck = null;
		//This isn't really a SocketID, just a unique ID that gets issued to new connections.
		private long m_nUDPNextUnusedSocketID = 1;
		//This is the unqiue ID for this message's guarenteed delivery.  The SocketID and the Message ID combined
		//should always be unique.
		private long m_nUDPNextUniqueGuarenteedMessageID = 1;

		//This is just a list of DSNetworkGuarenteedPacket objects
		private DSSortedList m_oUDPUnACKedGuarenteedPackets = new DSSortedList();
		//This is a list of lists, which are lists (by sender socketID) of IDs
		private DSSortedList m_oUDPACKedGuarenteedPackets = new DSSortedList();

        //Make sure we don't get any dupes due to UDP unguarenteed deliveries
        private DSNetworkPacketHistorys m_oUDPMessageHistory = new DSNetworkPacketHistorys();
		#endregion
 
		#region Delegate Variables
		private CallbackPlayerJoined		m_odDelegate_PlayerJoined;
		private CallbackPlayerQuit			m_odDelegate_PlayerQuit;
		private CallbackHostMigrated		m_odDelegate_HostMigrated;
		private CallbackDataReceived		m_odDelegate_DataReceived;
		private CallbackSessionTerminated	m_odDelegate_SessionTerminated;
		private CallbackConnection			m_odDelegate_Connection;
		private CallbackGameListUpdate		m_odDelegate_GameListUpdate;
		private CallbackCanceledGameSearch	m_odDelegate_CanceledGameSearch;
		private CallbackError				m_odDelegate_Error;

		public delegate void CallbackPlayerJoined(DSNetworkPlayer oPlayerJoined,bool bUs);
		public delegate void CallbackPlayerQuit(DSNetworkPlayer oPlayerQuit);
		public delegate void CallbackHostMigrated(DSNetworkPlayer oNewHost);
		public delegate void CallbackDataReceived( DSNetworkPlayer oSendingPlayer,DSNetworkPacket oPacket );
		public delegate void CallbackSessionTerminated(Microsoft.DirectX.DirectPlay.ResultCode nReason);
		public delegate void CallbackConnection( Microsoft.DirectX.DirectPlay.ResultCode oCode,string sDescription,bool bConnectionSuccessful );
		public delegate void CallbackGameListUpdate( FindHostsResponseInformation dpGameInfo );
		public delegate void CallbackCanceledGameSearch();
		private delegate void PeerCloseCallback(); // This delegate will be called when the session terminated event is fired.

		public delegate void CallbackError( System.Exception oError );
		#endregion
		#endregion


//we tag all messages with a new id but some are just acks and stuff, which is fine, if we aren't host.
//As host however, we send to all players so 50% is the best we can do in a 2 player game.  Make messageid
//assignment be player specific

		public DSNetworkWrapper()
		{
            m_oDebug = new DSBufferDebug();
            m_oDebug.DebugPath = DSMisc.GetDevelopmentAppPath() +
                            "DSNetworkWrapper_DebugFile_" +
                            DateTime.Now.ToString("yyyy-MM-dd hh") + "h" +
                            DateTime.Now.ToString("mm") + "m" +
                            DateTime.Now.ToString("ss") + "s" +
                            ".OUT";
		}
        public DSNetworkWrapper(string sDebugPath, bool bDebugMode)
        {
            m_oDebug = new DSBufferDebug();
            m_oDebug.DebugPath = sDebugPath;
            m_oDebug.DebugMode = bDebugMode;
        }
		public void Initialize( Guid oAppGuid,System.Net.Sockets.ProtocolType oProtocolType )
		{
			m_bInitialized = true;
			m_oProtocolType = oProtocolType;
			m_oAppGUID = oAppGuid;

			if( oProtocolType == System.Net.Sockets.ProtocolType.Tcp )
			{
				m_oPeerObject = new Peer();
				// First set up our event handlers (We only need events for the ones we care about)
				m_oPeerObject.PlayerCreated		+= new PlayerCreatedEventHandler(this.PlayerJoined);
				m_oPeerObject.PlayerDestroyed	+= new PlayerDestroyedEventHandler(this.PlayerQuit);
				m_oPeerObject.HostMigrated		+= new HostMigratedEventHandler(this.HostMigrated);
				m_oPeerObject.Receive			+= new ReceiveEventHandler(this.TCPIPDataReceived );
				m_oPeerObject.SessionTerminated += new SessionTerminatedEventHandler(this.SessionTerminated);
			}
		}
		public void Dispose()
		{
			// Cleanup DPlay
			if (m_oPeerObject != null)
				m_oPeerObject.Dispose();

			m_oPeerObject = null;

			m_oPlayerList.Clear();
		}

		#region TCPIP Specific Functions
		public virtual void HostGame( ServiceProviderInformation oServiceProvider, string sSessionName, string sMyPlayerName,int nPort )
		{
			if( m_bInitialized == false )
			{
				throw new System.Exception( "Network must first be initialized." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Udp )
			{
				throw new System.Exception( "ServiceProviderInformation is not a valid paramater for UDP." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Tcp )
			{
				PlayerInformation oMyInformation = new PlayerInformation();

				m_bIAmTheHost = true;
			
				//Start the pinging thread
				m_oPingThread = new Thread( new ThreadStart( this.PingLoop ) );
				m_oPingThread.IsBackground = true;
				m_oPingThread.Start();


				m_oOurAddress = new Address();
				m_oOurAddress.ServiceProvider = oServiceProvider.Guid;

				m_oAppDesc = new ApplicationDescription();
				m_oAppDesc.GuidApplication = m_oAppGUID;
				m_oAppDesc.SessionName = sSessionName;
			    
				m_oAppDesc.Flags = 0;
				m_oAppDesc.Flags |= SessionFlags.MigrateHost;
				m_oAppDesc.Flags |= SessionFlags.FastSigned;
				m_oAppDesc.Flags |= SessionFlags.NoDpnServer;

				m_oOurAddress.AddComponent( Address.KeyPort,nPort );
			
				oMyInformation.Name = sMyPlayerName;
				m_oPeerObject.SetPeerInformation( oMyInformation,SyncFlags.PeerInformation );
				m_oPeerObject.Host( m_oAppDesc,m_oOurAddress,HostFlags.OkToQueryForAddressing );

				m_bConnected = true;
			}
		}

		public void SearchForGames( ServiceProviderInformation oServiceProvider,string sIP,int nPort )
		{
			FindHostsFlags nFlags = 0;


			if( m_bInitialized == false )
			{
				throw new System.Exception( "Network must first be initialized." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Udp )
			{
				throw new System.Exception( "ServiceProviderInformation is not a valid paramater for UDP." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Tcp )
			{
				//Set up the event handlers
				m_oPeerObject.FindHostResponse += new FindHostResponseEventHandler( FindHostResponseMessageTCPIP );
				m_oPeerObject.ConnectComplete += new ConnectCompleteEventHandler(ConnectResult);
				m_oPeerObject.AsyncOperationComplete += new AsyncOperationCompleteEventHandler( GameSearchCompletedTCPIP );
 
				//Do something...
				m_oOurAddress = new Address();
				m_oOurAddress.ServiceProvider = oServiceProvider.Guid;

				//Get ourselves a host object
				if( m_oSearchingHostAddress != null )
				{
					m_oSearchingHostAddress.Dispose();
				}
				m_oSearchingHostAddress = new Address();
				m_oSearchingHostAddress.ServiceProvider = oServiceProvider.Guid;
				m_oSearchingHostAddress.AddComponent( Address.KeyHostname,sIP );
				m_oSearchingHostAddress.AddComponent( Address.KeyPort,nPort );

				//Time to enum our hosts
				m_oAppDesc = new ApplicationDescription();
				m_oAppDesc.GuidApplication = m_oAppGUID;
		    
				m_oPeerObject.FindHosts( m_oAppDesc,m_oSearchingHostAddress,m_oOurAddress,
									 null,Timeout.Infinite,0,Timeout.Infinite, nFlags, out m_nFindHostHandle);
				m_bSearchingForGames = true;
			}
		}

		public virtual void JoinGame( FindHostsResponseInformation oGame,string sMyPlayerName )
		{
			if( m_bInitialized == false )
			{
				throw new System.Exception( "Network must first be initialized." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Udp )
			{
				throw new System.Exception( "FindHostsResponseInformation is not a valid paramater for UDP." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Tcp )
			{
				PlayerInformation oMyInformation = new PlayerInformation();


				m_bIAmTheHost = false;
				oMyInformation.Name = sMyPlayerName;

				m_oPeerObject.SetPeerInformation( oMyInformation,SyncFlags.PeerInformation );
				m_oPeerObject.Connect( oGame.ApplicationDesc,oGame.sender,oGame.device,null,
								   out m_nConnectHandle,ConnectFlags.OkToQueryForAddressing );


				//Start the pinging thread
				m_oPingThread = new Thread( new ThreadStart( this.PingLoop ) );
				m_oPingThread.IsBackground = true;
				m_oPingThread.Start();


				//Set up our connect timer to check the results
				m_oTCPIPConnectingTimer = new System.Timers.Timer( 100 ); //A 100ms interval
				m_oTCPIPConnectingTimer.Elapsed += new System.Timers.ElapsedEventHandler( this.TCPIPConnectingTimer );
				//m_oTCPIPConnectingTimer.SynchronizingObject = (System.ComponentModel.ISynchronizeInvoke)this;
				m_oTCPIPConnectingTimer.Start();
			}
		}

		//Wait for a connect to complete
		private void TCPIPConnectingTimer( object sender, System.Timers.ElapsedEventArgs e )
		{
			if (m_oTCPIPConnectingTimer != null)
			{
				if( m_bConnected == true )
				{
					m_oTCPIPConnectingTimer.Stop();
					m_oTCPIPConnectingTimer = null;
				}
				else
				{
					//Need to set a timeout amount
					//Chris
					//MessageBox.Show(this,"Failed to connect.","Failed to connect: " + MSDXPlayResultCodeToString( m_nConnectResultCode ),MessageBoxButtons.OK,MessageBoxIcon.Information);
				}
			}
		}


		private void TCPIPDataReceived(object sender, ReceiveEventArgs e)
		{
			DSNetworkPacket oPacket = null;

			oPacket = new DSNetworkPacket( e );
			UDPInternalDataReceived( e.Message.SenderID,oPacket,null );

			e.Message.ReceiveData.Dispose(); // We no longer need the data, Dispose the buffer
		}
		

		//09/09/2005 Chris Hill  These functions are for the creation of a game.
		public ServiceProviderInformation[] GetServiceProviders()
		{
			if( m_oProtocolType != System.Net.Sockets.ProtocolType.Tcp )
			{
				throw new System.Exception( "GetServiceProviders is only valid for TCPIP." );
			}
			return( m_oPeerObject.GetServiceProviders( false ) );
		}
		public ServiceProviderInformation FindServiceProvider( string sGUID )
		{
			ServiceProviderInformation oRetVal = new ServiceProviderInformation();


			if( m_oProtocolType != System.Net.Sockets.ProtocolType.Tcp )
			{
				throw new System.Exception( "FindServiceProvider is only valid for TCPIP." );
			}

			foreach( ServiceProviderInformation oSPInfo in this.GetServiceProviders() )
			{
				if( oSPInfo.Guid.ToString().CompareTo( sGUID ) == 0 )
				{
					oRetVal = oSPInfo;
					break;
				}
			}

  
			return( oRetVal );
		}

		//Returns true if the given provider requires a port component
		//<param name="provider">ServiceProvider Guid</param>
		public bool ProviderRequiresPort( Guid provider )
		{
			if( m_oProtocolType != System.Net.Sockets.ProtocolType.Tcp )
			{
				throw new System.Exception( "ProviderRequiresPort is only valid for TCPIP." );
			}

			return( provider != Address.ServiceProviderSerial &&
					provider != Address.ServiceProviderModem &&
					provider != Address.ServiceProviderBlueTooth );
		}


		//A host was found and responded to our query
		private void FindHostResponseMessageTCPIP( object sender, FindHostResponseEventArgs dpMessage )
		{
			FindHostsResponseInformation dpInfo = new FindHostsResponseInformation();


			dpInfo.ApplicationDesc = dpMessage.Message.ApplicationDescription;
			dpInfo.device = dpMessage.Message.AddressDevice;
			dpInfo.sender = dpMessage.Message.AddressSender;
			dpInfo.RoundTripLatencyMs = dpMessage.Message.RoundTripLatencyMs;
			dpInfo.LastFoundTime = Environment.TickCount;

			GameListUpdate( dpInfo );
		}
		//An asynchronous operation was completed or cancelled
		private void GameSearchCompletedTCPIP( object sender, AsyncOperationCompleteEventArgs e )
		{
			if( e.Message.AsyncOperationHandle == m_nFindHostHandle )
			{
				m_bSearchingForGames = false;
				m_nFindHostHandle = 0;
				CanceledGameSearch();
			}
			else if( e.Message.AsyncOperationHandle == m_nConnectHandle )
			{
			}
		}



		//This routine walks the list of players and sets the new host flag
		private void UpdateTCPIPHostAfterMigration( long nSocketID )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.UpdateTCPIPHostAfterMigration";

			try
			{
				foreach( DSNetworkPlayer oNetPlayer in m_oPlayerList.GetValueList() )
				{
					oNetPlayer.IAmHost = ( oNetPlayer.SocketID == nSocketID );

					if( oNetPlayer.IAmHost == true )
					{
						m_oPlayerWhoIsHost = oNetPlayer;
					}
				}
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}

		private static string MSDXPlayResultCodeToString( Microsoft.DirectX.DirectPlay.ResultCode oCode )
		{
			string sRetVal = "";

			if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.ConnectionLost ){ sRetVal = "ConnectionLost"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.UserCancel ){ sRetVal = "UserCancel"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.TimedOut ){ sRetVal = "TimedOut"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.SessionFull ){ sRetVal = "SessionFull"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.PlayerNotReachable ){ sRetVal = "PlayerNotReachable"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.PlayerNotInGroup ){ sRetVal = "PlayerNotInGroup"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.PlayerLost ){ sRetVal = "PlayerLost"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.PlayerAlreadyInGroup ){ sRetVal = "PlayerAlreadyInGroup"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.NotReady ){ sRetVal = "NotReady"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.NotHost ){ sRetVal = "NotHost"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.NotAllowed ){ sRetVal = "NotAllowed"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.NoResponse ){ sRetVal = "NoResponse"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.NoHostPlayer ){ sRetVal = "NoHostPlayer"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.NoConnection ){ sRetVal = "NoConnection"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.HostTerminatedSession ){ sRetVal = "HostTerminatedSession"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.HostRejectedConnection ){ sRetVal = "HostRejectedConnection"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.DoesNotExist ){ sRetVal = "DoesNotExist"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.CannotCreatePlayer ){ sRetVal = "CannotCreatePlayer"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.CannotCreateGroup ){ sRetVal = "CannotCreateGroup"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.CannotCancel ){ sRetVal = "CannotCancel"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.Aborted ){ sRetVal = "Aborted"; }
			else if( oCode == Microsoft.DirectX.DirectPlay.ResultCode.Success ){ sRetVal = "Success"; }

			return( sRetVal );
		}
		
		#endregion
		#region UDP Specific Functions
		public virtual void HostGame( string sSessionName, string sMyPlayerName,int nPort )
		{
			DSNetworkPlayer oPlayer = null;


			if( m_bInitialized == false )
			{
				throw new System.Exception( "Network must first be initialized." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Udp )
			{
				m_bIAmTheHost = true;

				m_oUDPServer = new Socket( AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp );
				m_oIEndPoint = new IPEndPoint( IPAddress.Any,nPort );
				m_oUDPServer.Bind( m_oIEndPoint );


				//03/14/2007 Chris Hill  This turns off the failing of the ReceiveFrom call when you SentTo to a
				//client who is no longer available.
				//http://blog.devstone.com/Aaron/archive/2005/02/20/460.aspx
				const int SIO_UDP_CONNRESET = -1744830452;
				byte[] inValue = new byte[] { 0, 0, 0, 0 };     // == false
				byte[] outValue = new byte[] { 0, 0, 0, 0 };    // initialize to 0
				m_oUDPServer.IOControl(SIO_UDP_CONNRESET, inValue, outValue);

				oPlayer = AddPlayer( m_nUDPNextUnusedSocketID++,sMyPlayerName,true,true );

				StartUDPThreads();

				m_bConnected = true;
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Tcp )
			{
				throw new System.Exception( "ServiceProviderInformation is a required paramater for TCPIP." );
			}
		}
		public virtual void JoinGame( string sMyPlayerName,string sIP,int nPort )
		{
			if( m_bInitialized == false )
			{
				throw new System.Exception( "Network must first be initialized." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Udp )
			{
				m_bIAmTheHost = false;
				m_bConnected = true;
				m_sTempMyPlayerName = sMyPlayerName;

				//Establish our pseudo connection
				//m_oUDPClient = new UdpClient( sIP,nPort );

				m_oUDPClient = new Socket( AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp );
				m_oIEndPoint = new IPEndPoint( IPAddress.Parse( sIP ),nPort );

				//We have to connect in a thread so that we don't sit here and stall...
				m_oUDPClient.Connect( m_oIEndPoint );

				//03/14/2007 Chris Hill  This turns off the failing of the ReceiveFrom call when you SentTo to a
				//client who is no longer available.
				//http://blog.devstone.com/Aaron/archive/2005/02/20/460.aspx
				const int SIO_UDP_CONNRESET = -1744830452;
				byte[] inValue = new byte[] { 0, 0, 0, 0 };     // == false
				byte[] outValue = new byte[] { 0, 0, 0, 0 };    // initialize to 0
				m_oUDPClient.IOControl(SIO_UDP_CONNRESET, inValue, outValue);


				//Start the listening thread if it isn't already
				StartUDPThreads();

				//Tell them we want to join!
				m_sMyGUIDForJoining = DSMisc.GetGUID();
				DSNetworkPacket oNetPacket = new DSNetworkPacket( (int)enumSystemMessageTypes.UDPPlayerJoinRequst,
					m_oAppGUID.ToString(),
					m_sTempMyPlayerName,
					m_sMyGUIDForJoining );
				oNetPacket.SystemMessage = true;
				oNetPacket.GuarenteedDelivery = true;
				SendMsg( oNetPacket );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Tcp )
			{
				throw new System.Exception( "FindHostsResponseInformation is a required paramater for TCPIP." );
			}
		}
		private void StartUDPThreads()
		{
			if( m_oBlockedUDPDataReceiving != null )
			{
				m_oBlockedUDPDataReceiving.Abort();
				m_oBlockedUDPDataReceiving = null;
			}
			m_oBlockedUDPDataReceiving = new Thread( new ThreadStart( ThreadBlockedUDPDataReceive ) );
			m_oBlockedUDPDataReceiving.IsBackground = true;
			m_oBlockedUDPDataReceiving.Start();

			if( m_oUDP_ACKCheck != null )
			{
				m_oUDP_ACKCheck.Abort();
				m_oUDP_ACKCheck = null;
			}
			m_oUDP_ACKCheck = new Thread( new ThreadStart( ThreadUDP_ACKAndLivelenessCheck ) );
			m_oUDP_ACKCheck.IsBackground = true;
			m_oUDP_ACKCheck.Start();

			if( m_oPingThread != null )
			{
				m_oPingThread.Abort();
				m_oPingThread = null;
			}
			m_oPingThread = new Thread( new ThreadStart( this.PingLoop ) );
			m_oPingThread.IsBackground = true;
			m_oPingThread.Start();
		}
		public void StopObsoletePackets( long nMessageTypeToStopFor )
		{
            if (m_oPacketsToIgnoreIfObsolete.Contains(nMessageTypeToStopFor) == false)
            {
                m_oPacketsToIgnoreIfObsolete.Add(nMessageTypeToStopFor);
            }
		}

		private void ThreadBlockedUDPDataReceive()
		{ 
			DSNetworkPacket oPacket = null;
			IPEndPoint oIEndPt = new IPEndPoint(IPAddress.Any, 0); 
			EndPoint oEndPt = (EndPoint)oIEndPt;
			long nSocketID = 0;
			long nError = 0;
            

			while (true) 
			{ 
				try
				{
					IPEndPoint oIEndPt2 = new IPEndPoint(IPAddress.Any, 0);
					oEndPt = (EndPoint)oIEndPt2;
					if( m_oUDPServer != null )
					{
						//Means i'm the host
						nError = m_oUDPServer.ReceiveFrom( m_oaUDPMessageReceiveBuffer,ref oEndPt ); 
					}
					else if( m_oUDPClient != null )
					{
						nError = m_oUDPClient.ReceiveFrom( m_oaUDPMessageReceiveBuffer,ref oEndPt ); 
						//m_oaUDPMessageReceiveBuffer = m_oUDPClient.Receive( ref oIEndPt ); 
					}

					try	
					{
						oPacket = new DSNetworkPacket();
						oPacket.ParsePacket( m_oaUDPMessageReceiveBuffer );

						//Get this sender's SocketID
						nSocketID = oPacket.SocketIDSentFrom; 

                        //Make sure we haven't already received this packet
                        m_oDebug.WriteToDebug(1, "Check for dupe message: " + oPacket.MsgType.ToString() + ", " + oPacket.MessageID.ToString());

                        if (m_bIAmTheHost == true ||
                            oPacket.MessageID == 0 ||
                            m_oUDPMessageHistory.IsMessageADupe(oPacket.MsgType, oPacket.MessageID) == false)
                        {
                            m_oDebug.WriteToDebug(2, "Msg is not a dupe");

                            //Process the message
                            UDPInternalDataReceived(nSocketID, oPacket, oEndPt);
                        }
                        else
                        {
                            m_oDebug.WriteToDebug(2, "Msg is a dupe");
                        }
                        //while we are at it, clean out old records
                        m_oUDPMessageHistory.CleanOutOldMessages();
					}
					catch( System.Exception oEx )
					{
						m_oDebug.WriteToDebug( 1,"Error in DSInputWrapper Initialize... " + oEx.Message );
					}
				}
				catch( System.Net.Sockets.SocketException oEx )
				{
					//This means the player in question has been clipped
					if( oEx.ErrorCode == 10054 )
					{
						/*lock( m_oPlayerList )
						{
							oTempList = (DSSortedList)m_oPlayerList.Clone();
						}
						for( int i=0 ; i<oTempList.Count ; i++ )
						{
							oLoopPlayer = (DSNetworkPlayer)oTempList.GetByIndex( i );
							if( oLoopPlayer.UDPEndPoint == oEndPt )
							{
								//ToDo:  Remove the player
								break;
							}
						}*/
					}
					else
					{
						if( m_odDelegate_Error == null )
						{
							DSMisc.ShowErrors( new System.Exception( "ThreadBlockedUDPDataReceive Failed.",oEx ) );
						}
						else
						{
							m_odDelegate_Error( new System.Exception( "ThreadBlockedUDPDataReceive Failed.",oEx ) );
						}
					}
				}
				catch( System.Threading.ThreadAbortException )
				{
				}
				catch( System.Exception oEx )
				{
					if( m_odDelegate_Error == null )
					{
						DSMisc.ShowErrors( new System.Exception( "ThreadBlockedUDPDataReceive Failed.",oEx ) );
					}
					else
					{
						m_odDelegate_Error( new System.Exception( "ThreadBlockedUDPDataReceive Failed.",oEx ) );
					}
				}
			} 
		}
		private void ThreadUDP_ACKAndLivelenessCheck()
		{
			double nCurTime = 0;
			double nTicksPerSecond = DSMisc.GetQueryPerformanceFrequency();
            double nMsgSpan = 0, nJoinSpan = 0;
			DSSortedList oSocketIDACKList = null;
			DSSortedList oTempList = null;
			DSNetworkGuarenteedPacket oLoopGuarenteed = null;
			DSNetworkGuarenteedPacketACKsSent oLoopGuarenteedACKs = null;
			DSNetworkPlayer oLoopPlayer = null;
			DSNetworkPacket oNetPacket = null;
			

			while( true )
			{
				nCurTime = DSMisc.GetQueryPerformanceCounter();

 
				//Walk all the ACK's that we haven't gotten approved and resend them
				lock( m_oUDPUnACKedGuarenteedPackets )
				{
					for( int i=0 ; i<m_oUDPUnACKedGuarenteedPackets.Count ; i++ )
					{
						oLoopGuarenteed = (DSNetworkGuarenteedPacket)m_oUDPUnACKedGuarenteedPackets.GetByIndex( i );

                        if (oLoopGuarenteed != null && nCurTime - oLoopGuarenteed.TimeLastSent > nTicksPerSecond)
						{
							m_oDebug.WriteToDebug( 2,"Resend message " + oLoopGuarenteed.Packet.MessageID.ToString() + " of type " + oLoopGuarenteed.Packet.MsgType.ToString() + " due to no ACK" );
							ReSendMsg( oLoopGuarenteed.Packet );
							oLoopGuarenteed.TimeLastSent = DSMisc.GetQueryPerformanceCounter();;
						}
					}
				}

					  
				//Now walk the list of ACKs we've sent and any that are over x seconds old, remove.
				lock( m_oUDPACKedGuarenteedPackets )
				{
					for( int i=0 ; i<m_oUDPACKedGuarenteedPackets.Count ; i++ )
					{
						if( i >= m_oUDPACKedGuarenteedPackets.Count ){ break; }
						oSocketIDACKList = (DSSortedList)m_oUDPACKedGuarenteedPackets.GetByIndex( i );

						if( oSocketIDACKList.Count == 0 )
						{
							m_oUDPACKedGuarenteedPackets.RemoveAt( i );
							i--;
						}
						else
						{
							for( int n=0 ; n<oSocketIDACKList.Count ; n++ )
							{
								if( n >= oSocketIDACKList.Count ){ break; }
								oLoopGuarenteedACKs = (DSNetworkGuarenteedPacketACKsSent)oSocketIDACKList.GetByIndex( n );

								//This means its been over x seconds since we sent the last ACK with no request for another,
								//Its probabely safe to throw the record away at this point
                                if (oLoopGuarenteedACKs != null &&
                                    nCurTime - oLoopGuarenteedACKs.TimeLastSent > nTicksPerSecond * m_cSECONDS_SINCE_ACK_WITHNO_FURTHER_REQUESTS)
								{
									m_oDebug.WriteToDebug( 2,
											"Removing message " + oLoopGuarenteedACKs.MessageID.ToString() + 
											" from the list of obsolete messages" );
									oSocketIDACKList.RemoveAt( n );
									n--;
								}
							}
						}
					}
				}


				//Ok we'll also look for dead players here too
				lock( m_oPlayerList )
				{
					oTempList = m_oPlayerList.Clone();
				}
				for( int i=0 ; i<oTempList.Count ; i++ )
				{
					oLoopPlayer = (DSNetworkPlayer)oTempList.GetByIndex( i );

                    nMsgSpan = ( DSMisc.GetQueryPerformanceCounter() - oLoopPlayer.LastMsgTime ) / DSMisc.GetQueryPerformanceFrequency();
                    nJoinSpan = (DSMisc.GetQueryPerformanceCounter() - oLoopPlayer.JoinTime) / DSMisc.GetQueryPerformanceFrequency();

                    if ((oLoopPlayer.LastPingTime != 0 && nMsgSpan > m_nLivelenessTimoutTimeInSeconds) ||
                        (oLoopPlayer.LastPingTime == 0 && nJoinSpan > m_nLivelenessTimoutTimeInSeconds))
					{ 
						//We are the host
						if( m_oUs != null && m_oUs.IAmHost == true && oLoopPlayer.IsMe == false )
						{
							//Broadcast the player drop
							//Now tell everyone that this joker has quit
							oNetPacket = new DSNetworkPacket( (int)enumSystemMessageTypes.UDPPlayerHasQuit,oLoopPlayer.SocketID );
							oNetPacket.SystemMessage = true;
							oNetPacket.GuarenteedDelivery = true;
							SendMsg( oNetPacket );

                            m_oDebug.WriteToDebug(2,"Removing player " + oLoopPlayer.Name + " from the list due to ping failures");

							//Drop the dead player
							RemovePlayer( oLoopPlayer.SocketID );
						}
						//We are not the host
						else if( m_oUs != null && m_oUs.IAmHost == false && oLoopPlayer.IAmHost == true )
						{
							if( m_odDelegate_Connection != null )
							{
								m_odDelegate_Connection( Microsoft.DirectX.DirectPlay.ResultCode.ConnectionLost,"Connection to host has been lost",false );
							}								

							//Drop the dead player
							RemovePlayer( oLoopPlayer.SocketID );

							i--;
						}
					}
				}


				//Wait
				Thread.Sleep( 250 );
			}
		}
		private void UDPInternalDataReceived( long nSenderSocketID,DSNetworkPacket oPacket,EndPoint oUDPEndPt )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.UDPInternalDataReceived";
			byte[] oaByteArrayOfPacket = null;
			bool bSendMessageOn = false;
			bool bPacketIsObsolete = false;
			DSNetworkPlayer oNetPlayer = null;
			DSNetworkPacket oNetPacket = null;
			DSNetworkPlayer oLoopPlayer = null;
			DSNetworkPacket oACKPacket = null;
			DSSortedList oSocketIDACKList = null;
			DSNetworkGuarenteedPacketACKsSent oGuarenteedACK = null;
			DSSortedList oTempList = null;
int nLine = 0;

if (oPacket.MsgType == 11)
{
    nLine = 1;
}
			try
			{
				m_oDebug.WriteToDebug( 1,
					"Received message of type (" + oPacket.MsgType.ToString() + 
					"), ID (" + oPacket.MessageID.ToString() + ") System Msg(" + oPacket.SystemMessage + ") " +
					"FromSocketID(" + oPacket.SocketIDSentFrom.ToString() + ")" );

				if( m_oUs == null || nSenderSocketID != m_oUs.SocketID )
				{
					//Check to see if this is an obsolete packet... obsolete packets are packets that
					//were guarenteed for delivery, and were processed and ACKed.  If they arrive again,
					//say because the ACK didn't get delivered, we want to ignore the repeat message.
					if( oPacket.MessageID != 0 )
					{
						lock( m_oUDPACKedGuarenteedPackets )
						{
							//If this is an initial message before the player knows his own socket ID,
							//then the from field may be blank.  So its time to populate it for them.
							if( oPacket.MsgType == (int)enumSystemMessageTypes.UDPPlayerJoinRequst &&
								(string)oPacket.GetParamater( 0 ) == m_oAppGUID.ToString() &&
								m_oJoiningGUIDToSocketID.ContainsKey( (string)oPacket.GetParamater( 2 ) ) == true )
							{
								oPacket.SocketIDSentFrom = (long)m_oJoiningGUIDToSocketID.GetByKey( (string)oPacket.GetParamater( 2 ) );
							}

							oSocketIDACKList = (DSSortedList)m_oUDPACKedGuarenteedPackets.GetByKey( oPacket.SocketIDSentFrom );
						}
					}
					//Now, does this socket have this message as already processed?
					if( oPacket.MessageID != 0 && oSocketIDACKList != null &&
						oSocketIDACKList.ContainsKey( oPacket.MessageID ) == true )
					{
						//We totally ignore this message completely as if it never existed
						m_oDebug.WriteToDebug( 1,
							"MessageID " + oPacket.MessageID.ToString() + " has already been processed, ignoring." );

                        //Even if we don't trakc the historry of the message, record that we got one
                        //Mark this player as having sent a message back, so we don't loose connection...
                        oNetPlayer = GetPlayer(nSenderSocketID);
                        if (oNetPlayer != null)
                        {
                            oNetPlayer.LastMsgTime = DSMisc.GetQueryPerformanceCounter();
                        }
					}
					else
					{
                        //Record this traffic for a non-dupe message
                        //Mark this player as having sent a message back, so we don't loose connection...
                        oNetPlayer = GetPlayer(nSenderSocketID);
                        if (oNetPlayer != null)
                        {
                            oNetPlayer.LastMsgTime = DSMisc.GetQueryPerformanceCounter();
                            oNetPlayer.NumberOfMessagesRecieved++;
                            oNetPlayer.HighestMessageIDReceived = DSMisc.Max(oNetPlayer.HighestMessageIDReceived, oPacket.MessageID);
                        }

nLine = 1;
						//Now, process the message
						if( oPacket.SystemMessage == true )
						{
	nLine = 11;
							if( oPacket.MsgType == (int)enumSystemMessageTypes.UDP_ACK )
							{
								m_oDebug.WriteToDebug( 3,"Message is system ACK for MessageID " + oPacket.GetParamater( 0 ) );
	nLine = 12;
								if( m_oUs != null && 
									( oPacket.SocketIDToSendTo == 0 || oPacket.SocketIDToSendTo == m_oUs.SocketID ) )
								{
	nLine = 5;
									//Find our ACK list for this specific socket
									lock( m_oUDPUnACKedGuarenteedPackets )
									{
										if( m_oUDPUnACKedGuarenteedPackets.ContainsKey( (long)oPacket.GetParamater( 0 ) ) == true )
										{
											m_oDebug.WriteToDebug( 2,
												"Received ACK for " + oPacket.GetParamater( 0 ) );

											m_oUDPUnACKedGuarenteedPackets.Remove( (long)oPacket.GetParamater( 0 ) );
										}
										else
										{
											m_oDebug.WriteToDebug( 3,"MessageID " + oPacket.GetParamater( 0 ) + " is not in m_oUDPUnACKedGuarenteedPackets list." );
										}
									}
								}
								else
								{
									m_oDebug.WriteToDebug( 3,
											"Ignoring ACK1: USIsNull(" + ((bool)( m_oUs == null )).ToString() + ") " + 
											"SendSckID:(" + oPacket.SocketIDToSendTo.ToString() + ")" );
								}
							}
							else if( oPacket.MsgType == (int)enumSystemMessageTypes.UDPPlayerJoinRequst )
							{
								m_oDebug.WriteToDebug( 3,"Message is system UDP Join Request" );
nLine = 13;
								//We don't respond if they are looking for a different game... we woulden't want to 
								//give them false information.  
								if( (string)oPacket.GetParamater( 0 ) == m_oAppGUID.ToString() )
								{
									//New players joining have a GUID not a SocketID.  So lets give them one...
									if( m_oJoiningGUIDToSocketID.ContainsKey( (string)oPacket.GetParamater( 2 ) ) == true )
									{
										oPacket.SocketIDSentFrom = (long)m_oJoiningGUIDToSocketID.GetByKey( (string)oPacket.GetParamater( 2 ) );
									}
									//They don't have one yet so record our lookup
									else
									{
										m_oJoiningGUIDToSocketID.Add( (string)oPacket.GetParamater( 2 ),m_nUDPNextUnusedSocketID );
										oPacket.SocketIDSentFrom = m_nUDPNextUnusedSocketID;
									}								

									//Super-secretly add this player for us, so we can stash his end point
									DSNetworkPlayer oNewPlayer = AddPlayer( m_nUDPNextUnusedSocketID,(string)oPacket.GetParamater( 1 ),false,false );
									oNewPlayer.UDPEndPoint = oUDPEndPt;

									//Accept this new player!
									oNetPacket = new DSNetworkPacket( (int)enumSystemMessageTypes.UDPPlayerJoinReply,
																	true,
																	m_nUDPNextUnusedSocketID,
																	(string)oPacket.GetParamater( 2 ) );
									oNetPacket.SystemMessage = true;
									oNetPacket.GuarenteedDelivery = true;
									SendMsg( oNewPlayer,oNetPacket );

									//And tell this joker who joined before him, and tell the current people about the new guy
									lock( m_oPlayerList )
									{
										oTempList = (DSSortedList)m_oPlayerList.Clone();
									}
									for( int i=0 ; i<oTempList.Count ; i++ )
									{
										oLoopPlayer = (DSNetworkPlayer)oTempList.GetByIndex( i );

										//Make sure we don't tell him about himself, we did that already.  
										if( oLoopPlayer.SocketID != oNewPlayer.SocketID )
										{
                                            //Tell him about this person
											oNetPacket = new DSNetworkPacket( (int)enumSystemMessageTypes.UDPPlayerHasJoined,
																		oLoopPlayer.SocketID,oLoopPlayer.Name,oLoopPlayer.IAmHost );
											oNetPacket.SystemMessage = true;
											oNetPacket.GuarenteedDelivery = true;
											SendMsg( oNewPlayer,oNetPacket );

                                            //Now tell this person about our new addition, only don't tell me, I already know
                                            if (oLoopPlayer.IsMe == false)
                                            {
                                                oNetPacket = new DSNetworkPacket((int)enumSystemMessageTypes.UDPPlayerHasJoined,
                                                                            oNewPlayer.SocketID, oNewPlayer.Name, oNewPlayer.IAmHost);
                                                oNetPacket.SystemMessage = true;
                                                oNetPacket.GuarenteedDelivery = true;
                                                SendMsg(oLoopPlayer, oNetPacket);
                                            }
										}                                        
									}

									m_nUDPNextUnusedSocketID++;
								}
								else
								{
									m_oDebug.WriteToDebug( 3,"Join request is for GUID " + 
											(string)oPacket.GetParamater( 0 ) + " not our GUID of " + m_oAppGUID.ToString() );
								}

								m_oDebug.WriteToDebug( 4,"Done with player join request" );
							}
							else if( oPacket.MsgType == (int)enumSystemMessageTypes.UDPPlayerJoinReply )
							{
								DSNetworkPlayer oPlayer = null;
								int nSocketID = -1;

								m_oDebug.WriteToDebug( 3,"Message is system Player Join Reply" );
	nLine = 14;
								
								nSocketID = (int)(long)oPacket.GetParamater( 1 );
								oPlayer = AddPlayer( nSocketID,m_sTempMyPlayerName,false,true );
								if( (bool)oPacket.GetParamater( 0 ) == true && oPlayer != null )
								{
									//Tell me that it worked, but wait for the join message to process that
									Connection( Microsoft.DirectX.DirectPlay.ResultCode.Success,"",true ); 
								}
								else if( (bool)oPacket.GetParamater( 0 ) == false && oPlayer != null )
								{
									//Tell me that it failed
									Connection( Microsoft.DirectX.DirectPlay.ResultCode.HostRejectedConnection,"",false ); 
								}
							}
							else if( oPacket.MsgType == (int)enumSystemMessageTypes.UDPPlayerHasJoined )
							{
								m_oDebug.WriteToDebug( 3,"Message is system player (" + (string)oPacket.GetParamater( 1 ) + ") has Joined - SocketID: " + ((long)oPacket.GetParamater( 0 )).ToString());
	nLine = 16;
								AddPlayer( (long)oPacket.GetParamater( 0 ),(string)oPacket.GetParamater( 1 ),(bool)oPacket.GetParamater( 2 ),false );
							}
							else if( oPacket.MsgType == (int)enumSystemMessageTypes.UDPPlayerHasQuit )
							{
                                m_oDebug.WriteToDebug(3, "Message is system Player has quit - SocketID: " + ((long)oPacket.GetParamater(0)).ToString());
	nLine = 17;
								bSendMessageOn = true;
								RemovePlayer( (long)oPacket.GetParamater( 0 ) );
							}
							else if( oPacket.MsgType == (int)enumSystemMessageTypes.PingPacketRequest )
							{
								m_oDebug.WriteToDebug( 3,"Message is system Ping Request" );
	nLine = 18;
								oNetPacket = new DSNetworkPacket( (int)enumSystemMessageTypes.PingPacketReply,
																oPacket.GetParamater( 0 ) );
								oNetPacket.SystemMessage = true;
								SendMsg( oNetPacket );
							}
							else if( oPacket.MsgType == (int)enumSystemMessageTypes.PingPacketReply )
							{
                                m_oDebug.WriteToDebug(3, "Message is system Ping Reply from " + oPacket.GetParamater(0) );
	nLine = 19;
								if( Convert.ToInt32( oPacket.GetParamater( 0 ) ) == m_oUs.SocketID )
								{
									oNetPlayer = GetPlayer( nSenderSocketID );
									if( oNetPlayer != null )
									{
										oNetPlayer.PingStopCounter = DSMisc.GetQueryPerformanceCounter();
                                        oNetPlayer.LastPingTime = DSMisc.GetQueryPerformanceCounter();
										oNetPlayer.LastPingInSeconds = (double)( oNetPlayer.PingStopCounter - oNetPlayer.PingStartCounter ) / 
																	(double)DSMisc.GetQueryPerformanceFrequency();
									}
								}
							}
							else if( oPacket.MsgType == (int)enumSystemMessageTypes.UDPPlayerQuit )
							{
								m_oDebug.WriteToDebug( 3,"Message is system Player Quit" );
	nLine = 20;
								long nQuitingPlayerSocketID = (long)oPacket.GetParamater( 0 );
								DSNetworkPlayer oPlayerQuiting = null;							

								//Now tell everyone that this joker has quit
								oNetPacket = new DSNetworkPacket( (int)enumSystemMessageTypes.UDPPlayerHasQuit,
																nQuitingPlayerSocketID );
								oNetPacket.SystemMessage = true;
								oNetPacket.GuarenteedDelivery = true;
								SendMsg( oNetPacket );

								//Get our player
								if( m_odDelegate_PlayerQuit != null )
								{
									lock( m_oPlayerList )
									{
										oPlayerQuiting = (DSNetworkPlayer)m_oPlayerList.GetByKey( nQuitingPlayerSocketID );
									}
									m_odDelegate_PlayerQuit( oPlayerQuiting );
								}

								//And tell this joker who joined before him
								lock( m_oPlayerList )
								{
									m_oPlayerList.Remove( nQuitingPlayerSocketID );
								}

								//And remove the player from our internal ack tracking.  
								m_oUDPACKedGuarenteedPackets.Remove( oPacket.SocketIDSentFrom );
							}
	nLine = 15;
						}
						else 
						{
							m_oDebug.WriteToDebug( 3,"Message is not a system message" );
	nLine = 7;

                            //First of all, is this packet obsolete and do we care?
                            if (oNetPlayer != null && m_oPacketsToIgnoreIfObsolete.Contains(oPacket.MsgType) == true)
                            {
                                bPacketIsObsolete = oNetPlayer.ObsoletePacketTracker.IsPacketObsolete(oPacket.MsgType, oPacket.MessageID);
                            }

if (oPacket.MsgType == 11)
{
nLine = 7;
}

	nLine = 8;
							if( bPacketIsObsolete == false )
							{
								//If its for us then process it
                                if (oPacket.SocketIDToSendTo == 0 || (m_oUs != null && oPacket.SocketIDToSendTo == m_oUs.SocketID))
                                {
                                    oNetPlayer = GetPlayer(nSenderSocketID);
                                    if (oNetPlayer != null)
                                    {
                                        //Now that we know that the packet is for us, mark it as having been received
                                        oNetPlayer.ObsoletePacketTracker.RecordPacket(oPacket.MsgType, oPacket.MessageID);

                                        try
                                        {
                                            DataReceived(oNetPlayer, oPacket);
                                        }
                                        catch (System.Exception oEx)
                                        {
                                            m_oDebug.WriteToDebug(1, "UDPInternalDataReceived Error: " + oEx.Message);
                                            //DSMisc.ShowErrors( oEx );
                                        }
                                    }
                                    else
                                    {
                                        //Throw an error?  We received a message from a player who
                                        //is not in our list... for a non-system message this should
                                        //not be possible.
                                        bSendMessageOn = false;
                                        m_oDebug.WriteToDebug(2, "Rejecting packet of type " + oPacket.MsgType.ToString() + " (" + ((int)oPacket.MsgType).ToString() + ") because sender doesn't exist yet.");
                                    }
                                }
                                else
                                {
                                    if (m_oUs == null)
                                    {
                                        m_oDebug.WriteToDebug(3, "Packet is not for us: SendTo(" + oPacket.SocketIDToSendTo.ToString() + ") " +
                                                                 "MySocketID(Null) ");
                                    }
                                    else
                                    {
                                        m_oDebug.WriteToDebug(3, "Packet is not for us: SendTo(" + oPacket.SocketIDToSendTo.ToString() + ") " +
                                                                 "MySocketID(" + m_oUs.SocketID.ToString() + ") ");
                                    }
                                }

	nLine = 9;
								//Send this onto everyone else if this is UDP and we are the host and it was intended for
								//more people than just us
								if( oNetPlayer != null && 
									m_oProtocolType == System.Net.Sockets.ProtocolType.Udp && 
									( oPacket.SocketIDToSendTo == 0 || 
                                        ( m_oUs != null && oPacket.SocketIDToSendTo != m_oUs.SocketID ) ) )
								{
									bSendMessageOn = true;
								}
							}
                            else
                            {
                                m_oDebug.WriteToDebug(3, "Packet is obsolete");
                            }
						}


						//Send it on?
						if( bSendMessageOn == true && m_oUs != null && m_oUs.IAmHost == true )
						{
							m_oDebug.WriteToDebug( 4,"Send the message on" );

							oaByteArrayOfPacket = oPacket.GetPacketByteArray();
							for( int nPlayerIndex=0 ; nPlayerIndex<m_oPlayerList.Count ; nPlayerIndex++ )
							{
								oLoopPlayer = (DSNetworkPlayer)m_oPlayerList.GetByIndex( nPlayerIndex );
								if( nSenderSocketID != oLoopPlayer.SocketID &&
									oLoopPlayer.SocketID != m_oUs.SocketID && 
									( oPacket.SocketIDToSendTo == 0 || oPacket.SocketIDToSendTo == oLoopPlayer.SocketID ) )
								{
									m_oUDPServer.SendTo( oaByteArrayOfPacket,oaByteArrayOfPacket.Length,
										SocketFlags.None,oLoopPlayer.UDPEndPoint );
								}
							}
						}
					}
nLine = 10;


					//Ok, was this a guarenteed message?
					if( oPacket.GuarenteedDelivery == true && 
						( 
							m_oUs != null &&
							( oPacket.SocketIDToSendTo == 0 || oPacket.SocketIDToSendTo == m_oUs.SocketID )
						) 
						||
						( 
							m_oUs == null &&
							oPacket.MsgType == (long)enumSystemMessageTypes.UDPPlayerJoinReply &&
							m_sMyGUIDForJoining == (string)oPacket.GetParamater( 2 )
						)
					  )
					{
						m_oDebug.WriteToDebug( 2,
							"Sending guarenteed message ACK for type (" + oPacket.MsgType.ToString() + 
							") needing an ACK from player " + nSenderSocketID.ToString() + 
							" with a MessageID of message " + oPacket.MessageID.ToString() );

						//ACK this message
						oACKPacket = new DSNetworkPacket( (int)enumSystemMessageTypes.UDP_ACK,oPacket.MessageID );
						oACKPacket.SocketIDToSendTo = oPacket.SocketIDSentFrom;
						oACKPacket.SystemMessage = true;
						SendMsg( oACKPacket );
nLine = 2;
						//Find our ACK list for this specific socket
						oSocketIDACKList = (DSSortedList)m_oUDPACKedGuarenteedPackets.GetByKey( oPacket.SocketIDSentFrom );
						if( oSocketIDACKList == null )
						{
							oSocketIDACKList = new DSSortedList();
							m_oUDPACKedGuarenteedPackets.Add( oPacket.SocketIDSentFrom,oSocketIDACKList );
							m_oDebug.WriteToDebug( 3,"Creating SocketID storage list for " + oPacket.SocketIDSentFrom.ToString() );
						}
nLine = 3;
						//Have we already ACK'd this message?  We still ACK it, since the receiver didn't get the
						//ACK, but we also don't process it again.
						if( oSocketIDACKList.ContainsKey( oPacket.MessageID ) == false )
						{
							oGuarenteedACK = new DSNetworkGuarenteedPacketACKsSent();
							oGuarenteedACK.MessageID = oPacket.MessageID;
							oGuarenteedACK.TimeLastSent = DSMisc.GetQueryPerformanceCounter();
							//Save that we ACK'ed this message
							oSocketIDACKList.Add( oPacket.MessageID,oGuarenteedACK );

							m_oDebug.WriteToDebug( 3,
													 "Adding message ID to socket list" );
						}
						else
						{
							m_oDebug.WriteToDebug( 3,
													 "Message ID was already in the list" );
						}
					}
					//Not guarenteed
					else
					{
						m_oDebug.WriteToDebug( 2,
							"Message is not guarenteed Type(" + oPacket.MsgType.ToString() + ") " + 
							"Guarented(" + oPacket.GuarenteedDelivery.ToString() + ") " + 
							"UsIsNull(" + ((bool)( m_oUs == null )).ToString() + ") " +
							"SendToSocketID(" + oPacket.SocketIDToSendTo.ToString() + ") " );
					}
				}
                else
                {
                    bool bUsIsNull = (m_oUs == null);
                    if( m_oUs == null )
                    {
                        m_oDebug.WriteToDebug(2, "Ignoring Message: Us(" + bUsIsNull.ToString() + ") " + 
                                                 "SenderSocketID(" + nSenderSocketID.ToString() + ") MySocketID(Null)" );
                    }
                    else
                    {
                        m_oDebug.WriteToDebug(2, "Ignoring Message: Us(" + bUsIsNull.ToString() + ") " + 
                                                 "SenderSocketID(" + nSenderSocketID.ToString() + ") MySocketID(" + m_oUs.SocketID.ToString() + ")" );
                    }
                }
			}
			catch( System.Exception oEx )
			{
				m_oDebug.WriteToDebug( 2,
					"Error: " + oEx.Message + " (" + "Scott's special box - " + nLine.ToString() + ")" );

MessageBox.Show( oEx.StackTrace,"Scott's special box - " + nLine.ToString() );
				long nType = -1;
				if( oPacket != null )
				{
					nType = oPacket.MsgType;
				}
				if( m_odDelegate_Error == null )
				{
					DSMisc.ShowErrors( new System.Exception( sRoutineName + " Failed on packet type " + nType.ToString() + ".",oEx ) );
				}
				else
				{
					m_odDelegate_Error( new System.Exception( sRoutineName + " Failed on packet type " + nType.ToString() + ".",oEx ) );
				}
			}
		}
		#endregion
		
		public virtual void QuitGame()
		{
			if( m_bInitialized == false )
			{
				throw new System.Exception( "Network must first be initialized." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Tcp )
			{
				m_oPingThread.Abort();
				m_oPingThread = null;

				if( m_bSearchingForGames == true )
				{
					CancelGameSearch();
				}
				if( m_nConnectHandle != 0 )
				{
					CancelGameJoin();
				}

				m_oPeerObject.TerminateSession( null );
			}
			//If we have a host, send a quit message
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Udp )
			{
				//Tell them we want to quit
				if( m_oUs != null )
				{
					DSNetworkPacket oNetPacket = new DSNetworkPacket( (int)enumSystemMessageTypes.UDPPlayerQuit,m_oUs.SocketID );
					oNetPacket.SystemMessage = true;
					//No sence in making it guarenteed, we won't be checking for re-transmissions
					//oNetPacket.GuarenteedDelivery = true;
					SendMsg( oNetPacket );
				}


				//Stop all our threads
				if( m_oBlockedUDPDataReceiving != null )
				{
					m_oBlockedUDPDataReceiving.Abort();
					m_oBlockedUDPDataReceiving = null;
				}
				if( m_oUDP_ACKCheck != null )
				{
					m_oUDP_ACKCheck.Abort();
					m_oUDP_ACKCheck = null;
				}
				if( m_oPingThread != null )
				{
					m_oPingThread.Abort();
					m_oPingThread = null;
				}


				//Shut it all down
				m_bConnected = false;

				m_oUDPClient.Close();
				m_oUDPClient = null;
				m_oIEndPoint = null;
			}
		}
		public void CancelGameSearch()
		{
			if( m_bSearchingForGames == false )
			{
				throw new System.Exception( "No game search currently in progress." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Udp )
			{
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Tcp )
			{
				if( m_nFindHostHandle != 0 )
				{
					m_oPeerObject.CancelAsyncOperation( m_nFindHostHandle );
				}
			}

			m_bSearchingForGames = false;
		}
		public void CancelGameJoin()
		{
			if( m_nConnectHandle != 0 )
			{
				m_oPeerObject.CancelAsyncOperation( m_nConnectHandle );
			}
		}
		public void PingAllPlayers()
		{
			DSNetworkPacket oNetPacket = null;
			DSNetworkPlayer oLoopPlayer = null;
			long nStartCounter = 0;


			if( m_oUs != null && m_oPlayerList != null )
			{
				nStartCounter = DSMisc.GetQueryPerformanceCounter();

				for( int nPlrIndex=0 ; nPlrIndex<m_oPlayerList.Count ; nPlrIndex++ )
				{
					oLoopPlayer = (DSNetworkPlayer)m_oPlayerList.GetByIndex( nPlrIndex );

					oLoopPlayer.PingStartCounter = nStartCounter;
					oLoopPlayer.PingStopCounter = 0;
				}

				oNetPacket = new DSNetworkPacket( (int)enumSystemMessageTypes.PingPacketRequest,m_oUs.SocketID.ToString() );
				oNetPacket.SystemMessage = true;
				SendMsg( oNetPacket );

                m_oDebug.WriteToDebug(2,"Sending ping request to all");
			}
		}
		public void PingLoop()
		{
			while( true )
			{
				//Delay, woulden't want this thread to spin to much
				if( m_nSecondsBetweenPings == 0 )
				{
					Thread.Sleep( 15 * 1000 );
				}
				else
				{
					Thread.Sleep( (int)( m_nSecondsBetweenPings * 1000 ) );
				}

                //Send out a ping
                m_oDebug.WriteToDebug(2, "PingAllPlayers");
                PingAllPlayers();
			}
		}

		//Fired when a connect complete message is received from DirectPlay.  Fire our event to notify the sample we've connected
		private void ConnectResult(object sender, ConnectCompleteEventArgs e)
		{
			m_bConnected = ( e.Message.ResultCode == 0 );
			Connection( e.Message.ResultCode,MSDXPlayResultCodeToString( e.Message.ResultCode ),m_bConnected );
		}
		public DSNetworkPlayer GetPlayer( long nSocketID )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.GetPlayer";
			DSNetworkPlayer oRetVal = null;

			try
			{
				oRetVal = (DSNetworkPlayer)m_oPlayerList.GetByKey( nSocketID );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}

			return( oRetVal );
		}
		private DSNetworkPlayer AddPlayer( long nSocketID,string sPlayerName,bool bIsHost,bool bIsMe )
		{
			bool bAdded = false;
			DSNetworkPlayer oPlayer = null;


			//We lock the data here since it is shared across multiple threads.
			lock( m_oPlayerList )
			{
				if( m_oPlayerList.ContainsKey( nSocketID ) == false )
				{
					oPlayer = new DSNetworkPlayer();
					m_oPlayerList.Add( nSocketID,oPlayer );
					bAdded = true;
				}
			}

			if( bAdded == true )
			{
                oPlayer.JoinTime = DSMisc.GetQueryPerformanceCounter();
				oPlayer.SocketID = nSocketID;
				oPlayer.Name = sPlayerName;
				oPlayer.IAmHost = bIsHost;
				oPlayer.IsMe = bIsMe;//( m_nSocketIDWeWillBeGivenOnJoinForUDP == oPlayer.SocketID );

				//Save this player id if it's ourselves
				if( oPlayer.IsMe == true )
				{
					m_oUs = oPlayer;
				}
				if( m_odDelegate_PlayerJoined != null )
				{
					m_odDelegate_PlayerJoined( oPlayer,oPlayer.IsMe );
				}
				if( oPlayer.IAmHost == true )
				{
					m_oPlayerWhoIsHost = oPlayer;
				}
			}


			return( oPlayer );
		}
		private void RemovePlayer( long nSocketID )
		{
			DSNetworkPlayer oPlayerQuiting = null;
			DSSortedList oTempList = null;
			bool bRemoveAll = false;


			//We lock the data here since it is shared across multiple threads.
			lock( m_oPlayerList )
			{
				oPlayerQuiting = (DSNetworkPlayer)m_oPlayerList.GetByKey( nSocketID );
				m_oPlayerList.Remove( nSocketID );
			}

			if( oPlayerQuiting != null )
			{
				//Save this player id if it's ourselves
				if( m_oUs != null && m_oUs.SocketID == nSocketID )
				{
					m_oUs = null;
					bRemoveAll = true;
				}
				if( m_oPlayerWhoIsHost != null && m_oPlayerWhoIsHost.SocketID == nSocketID )
				{
					m_oPlayerWhoIsHost = null;
					bRemoveAll = true;
				}

				if( m_odDelegate_PlayerQuit != null )
				{
					m_odDelegate_PlayerQuit( oPlayerQuiting );
				}
				if( bRemoveAll == true )
				{
					lock( m_oPlayerList )
					{
						oTempList = (DSSortedList)m_oPlayerList.Clone();
					}
					for( int i=0 ; i<oTempList.Count ; i++ )
					{
						RemovePlayer( (long)oTempList.GetKey( i ) );
					}
				}
			}
		}
		public void CloseNetwork()
		{
			if( m_bInitialized == false )
			{
				throw new System.Exception( "Network must first be initialized to be closed." );
			}
			else if( m_bConnected == false )
			{
				throw new System.Exception( "Network must first be connected to be closed." );
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Udp )
			{
			}
			else if( m_oProtocolType == System.Net.Sockets.ProtocolType.Tcp )
			{
				m_oPeerObject.TerminateSession( null );
			}

			m_bConnected = false;
		}

		#region SendMsg Functions
        private void ReSendMsg(DSNetworkPacket oPacket)
        {
            SendMsg(false, null, oPacket);
        }

        private void SendMsg(bool bAlwaysAssignID,DSNetworkPlayer oTargetPlayer, DSNetworkPacket oPacket)
        {
            const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
            byte[] oaByteArrayOfPacket = null;
            int nError = 0;
            DSSortedList oTempList = null;
            DSNetworkPlayer oLoopPlayer = null;
            DSNetworkGuarenteedPacket oGuarentee = null;


            try
            {
                if (oPacket.MessageID == 0 || bAlwaysAssignID == true )
                {
                    oPacket.MessageID = m_nUDPNextUniqueGuarenteedMessageID++;
                }

                m_oDebug.WriteToDebug(2,
                        "SendMsg: Type(" + oPacket.MsgType + ") ID(" + oPacket.MessageID.ToString() +
                        ", Only if guarenteed, Guarentted = " + oPacket.GuarenteedDelivery.ToString() +
                        ") System Msg(" + oPacket.SystemMessage + ") SendTo:" + oPacket.SocketIDToSendTo.ToString());

                if (m_bConnected == true)
                {
                    // Now we've got the data setup, send it off.
                    if (m_oProtocolType == System.Net.Sockets.ProtocolType.Udp)
                    {
                        //Assign who we sent this to and from
                        if (m_oUs != null)
                        {
                            oPacket.SocketIDSentFrom = m_oUs.SocketID;
                        }
                        else if (oPacket.SystemMessage == false)
                        {
                            m_oDebug.WriteToDebug(4, "Unable to send message of type " + oPacket.MsgType.ToString() + " because we have no self identifier.");
                            throw new System.Exception("Unable to send message of type " + oPacket.MsgType.ToString() + " because we have no self identifier.");
                        }
                        if (oTargetPlayer != null)
                        {
                            m_oDebug.WriteToDebug(3, "The target player(2) is " + oTargetPlayer.Name);
                            oPacket.SocketIDToSendTo = oTargetPlayer.SocketID;
                        }
                        else if (oPacket.SocketIDToSendTo != 0)
                        {
                            oTargetPlayer = (DSNetworkPlayer)m_oPlayerList.GetByKey(oPacket.SocketIDToSendTo);
                            if (oTargetPlayer == null)
                            {
                                m_oDebug.WriteToDebug(3, "The target player(2) is SocketID " + oPacket.SocketIDToSendTo.ToString());
                            }
                            else
                            {
                                m_oDebug.WriteToDebug(3, "The target player(2) is " + oTargetPlayer.Name);
                            }
                        }
                        else
                        {
                            m_oDebug.WriteToDebug(3, "The target player is null");
                            if (oPacket.MsgType == 6)
                            {
                                m_oDebug.WriteToDebug(3, "Can't ACK!!");
                            }
                        }

                        //Is this a guarenteed packet?  Then tag it
                        if (oPacket.GuarenteedDelivery == true && //oPacket.MessageID == 0 &&
                            (m_oUs == null || oPacket.SocketIDSentFrom == m_oUs.SocketID))
                        {
                            m_oDebug.WriteToDebug(3,
                                "Add MessageID " + m_nUDPNextUniqueGuarenteedMessageID.ToString() +
                                " ACK request since this is a guarenteed packet");

                            //01/20/2008 Chris Hill  Always send a unique MssageID now.
                            //Save the packet for resend later
                            //oPacket.MessageID = m_nUDPNextUniqueGuarenteedMessageID++;

                            oGuarentee = new DSNetworkGuarenteedPacket();
                            oGuarentee.Packet = oPacket;
                            oGuarentee.TimeLastSent = DSMisc.GetQueryPerformanceCounter();

                            if (m_oUDPUnACKedGuarenteedPackets.ContainsKey(oPacket.MessageID) == false)
                            {
                                m_oUDPUnACKedGuarenteedPackets.Add(oPacket.MessageID, oGuarentee);
                            }
                        }
                        else
                        {
                            m_oDebug.WriteToDebug(3,
                                "Message not guarenteed: Guarenteed(" + oPacket.GuarenteedDelivery.ToString() + ") " +
                                "CrntMsgID(" + oPacket.MessageID.ToString() + ") " +
                                "UsIsNull(" + ((bool)(m_oUs == null)).ToString() + ") " +
                                "ScktIDFrm(" + oPacket.SocketIDSentFrom.ToString() + ")");
                        }

                        //Now send our packet
                        oaByteArrayOfPacket = oPacket.GetPacketByteArray();

                        //Is our stream to big?
                        if (oaByteArrayOfPacket.Length > 60000)
                        {
                            m_oDebug.WriteToDebug(3, "Error, stream to big!");
                            throw new System.Exception("Error, stream to big!");
                        }

                        if (this.m_bIAmTheHost == true)
                        {
                            m_oDebug.WriteToDebug(3, "Sending from host");

                            lock (m_oPlayerList)
                            {
                                oTempList = (DSSortedList)m_oPlayerList.Clone();
                            }
                            for (int nPlayerIndex = 0; nPlayerIndex < oTempList.Count; nPlayerIndex++)
                            {
                                oLoopPlayer = (DSNetworkPlayer)oTempList.GetByIndex(nPlayerIndex);
                                if (oLoopPlayer.IsMe == false && oLoopPlayer.UDPEndPoint != null &&
                                    (oTargetPlayer == null || oTargetPlayer.SocketID == oLoopPlayer.SocketID))
                                {
                                    nError = m_oUDPServer.SendTo(oaByteArrayOfPacket, oaByteArrayOfPacket.Length,
                                                                    SocketFlags.None, oLoopPlayer.UDPEndPoint);

                                    m_oDebug.WriteToDebug(4, "Sent to " + oLoopPlayer.Name + " with error code of " + nError.ToString());
                                }
                                else
                                {
                                    m_oDebug.WriteToDebug(4, "Did not send to " + oLoopPlayer.Name + ".  IsMe:" + oLoopPlayer.IsMe.ToString() + " PlrSocketID:" + oLoopPlayer.SocketID.ToString());
                                    if (oLoopPlayer.UDPEndPoint == null)
                                    {
                                        m_oDebug.WriteToDebug(5, "UDP End Point is null");
                                    }
                                    if (oTargetPlayer == null)
                                    {
                                        m_oDebug.WriteToDebug(5, "Target Player End Point is null");
                                    }
                                    else
                                    {
                                        m_oDebug.WriteToDebug(5, "Target Player SocketID:" + oTargetPlayer.SocketID.ToString());
                                    }
                                }
                            }
                        }
                        else
                        {
                            m_oDebug.WriteToDebug(3, "Sending from client");

                            nError = m_oUDPClient.SendTo(oaByteArrayOfPacket, oaByteArrayOfPacket.Length,
                                                          SocketFlags.None, m_oIEndPoint);

                            m_oDebug.WriteToDebug(3, "Done Sending, nError = " + nError.ToString());
                        }
                    }
                    else if (m_oProtocolType == System.Net.Sockets.ProtocolType.Tcp)
                    {
                        try
                        {
                            if (this.IsInitialized == false)
                            {
                                throw new System.Exception("No network connections were found.  Peer object is closed.");
                            }
                            else if (oTargetPlayer == null)
                            {
                                m_oPeerObject.SendTo((int)PlayerID.AllPlayers, oPacket.GetPacketDX(), 0, SendFlags.Guaranteed);
                            }
                            else
                            {
                                m_oPeerObject.SendTo((int)oTargetPlayer.SocketID, oPacket.GetPacketDX(), 0, SendFlags.Guaranteed);
                            }
                        }
                        catch (System.Exception oEx)
                        {
                            DSMisc.ShowErrors(oEx);
                        }
                    }
                }
                else
                {
                    m_oDebug.WriteToDebug(3, "Unable to send a message before the connection is established.");
                    throw new System.Exception("Unable to send a message before the connection is established.");
                }
            }
            catch (System.Exception oEx)
            {
                 throw new System.Exception(sRoutineName + " Failed.", oEx);
            }
        }

		public void SendMsg( DSNetworkPlayer oTargetPlayer,DSNetworkPacket oPacket )
		{
            SendMsg(true, oTargetPlayer, oPacket);
		}
		public void SendMsg( DSNetworkPacket oPacket )
		{
			SendMsg( true,null,oPacket );
		}
		public void SendMsg( long nMsgType )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType );
				SendMsg( oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( long nMsgType,object oParam1 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType,oParam1 );
				SendMsg( oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( long nMsgType,object oParam1,object oParam2 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType,oParam1,oParam2 );
				SendMsg( oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( long nMsgType,object oParam1,object oParam2,object oParam3 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DarkStrideToolbox.DSNetworkPacket( nMsgType,oParam1,oParam2,oParam3 );
				SendMsg( oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( long nMsgType,object oParam1,object oParam2,object oParam3,object oParam4 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType,oParam1,oParam2,oParam3,oParam4 );
				SendMsg( oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( long nMsgType,object oParam1,object oParam2,object oParam3,object oParam4,object oParam5 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType,oParam1,oParam2,oParam3,oParam4,oParam5 );
				SendMsg( oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}

		public void SendMsg( DSNetworkPlayer oTargetPlayer,long nMsgType )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType );
				SendMsg( oTargetPlayer,oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( DSNetworkPlayer oTargetPlayer,long nMsgType,object oParam1 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType,oParam1 );
				SendMsg( oTargetPlayer,oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( DSNetworkPlayer oTargetPlayer,long nMsgType,object oParam1,object oParam2 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType,oParam1,oParam2 );
				SendMsg( oTargetPlayer,oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( DSNetworkPlayer oTargetPlayer,long nMsgType,object oParam1,object oParam2,object oParam3 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DarkStrideToolbox.DSNetworkPacket( nMsgType,oParam1,oParam2,oParam3 );
				SendMsg( oTargetPlayer,oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( DSNetworkPlayer oTargetPlayer,long nMsgType,object oParam1,object oParam2,object oParam3,object oParam4 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType,oParam1,oParam2,oParam3,oParam4 );
				SendMsg( oTargetPlayer,oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void SendMsg( DSNetworkPlayer oTargetPlayer,long nMsgType,object oParam1,object oParam2,object oParam3,object oParam4,object oParam5 )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SendMsg";
			DSNetworkPacket oPacket = null;

			try
			{
				oPacket = new DSNetworkPacket( nMsgType,oParam1,oParam2,oParam3,oParam4,oParam5 );
				SendMsg( oTargetPlayer,oPacket );
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}

		#endregion
		#region Delegate Functions
		public virtual void PlayerJoined(object sender, PlayerCreatedEventArgs e)
		{
			const string sRoutineName = "DarkStrideToolbox.Network.PlayerJoined";			
			PlayerInformation peerInfo;
			DSNetworkPlayer oPlayer = null;

			try
			{
				peerInfo = m_oPeerObject.GetPeerInformation(e.Message.PlayerID);
				oPlayer = new DSNetworkPlayer();
				oPlayer.SocketID = e.Message.PlayerID;
				oPlayer.Name = peerInfo.Name;
				oPlayer.IAmHost = peerInfo.Host;
				oPlayer.IsMe = peerInfo.Local;

				//We lock the data here since it is shared across multiple threads.
				lock (m_oPlayerList)
				{
					m_oPlayerList.Add( oPlayer.SocketID,oPlayer );
				}

				//Save this player id if it's ourselves
				if( peerInfo.Local == true )
				{
					m_oUs = oPlayer;
				}
				if( m_odDelegate_PlayerJoined != null )
				{
					m_odDelegate_PlayerJoined( oPlayer,peerInfo.Local );
				}
				if( oPlayer.IAmHost == true )
				{
					m_oPlayerWhoIsHost = oPlayer;
				}
			}
			catch( System.Exception oEx )
			{
				if( m_odDelegate_Error == null )
				{
					DSMisc.ShowErrors( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
				else
				{
					m_odDelegate_Error( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
			}
		}

		public virtual void PlayerQuit(object sender, PlayerDestroyedEventArgs e)
		{
			const string sRoutineName = "DarkStrideToolbox.Network.PlayerQuit";
			DSNetworkPlayer oPlayer = null;

			try
			{
				// Remove this player from our list
				// We lock the data here since it is shared across multiple threads.
				lock (m_oPlayerList)
				{
					oPlayer = GetPlayer(e.Message.PlayerID);
					m_oPlayerList.Remove( oPlayer.SocketID );

					// Update our number of player and our button
					//m_oTest.SetLabelText( m_oPlayerList.Count.ToString() );
				}

				if( m_odDelegate_PlayerQuit != null )
				{
					m_odDelegate_PlayerQuit( oPlayer );
				}
			}
			catch( System.Exception oEx )
			{
				if( m_odDelegate_Error == null )
				{
					DSMisc.ShowErrors( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
				else
				{
					m_odDelegate_Error( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
			}
		}

		public virtual void HostMigrated(object sender, HostMigratedEventArgs e)
		{
			const string sRoutineName = "DarkStrideToolbox.Network.HostMigrated";
			DSNetworkPlayer oPlayer = null;

			try
			{
				UpdateTCPIPHostAfterMigration( e.Message.NewHostID );

				oPlayer = GetPlayer( e.Message.NewHostID );

				if( m_odDelegate_HostMigrated != null )
				{
					m_odDelegate_HostMigrated( oPlayer );
				}
			}
			catch( System.Exception oEx )
			{
				if( m_odDelegate_Error == null )
				{
					DSMisc.ShowErrors( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
				else
				{
					m_odDelegate_Error( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
			}
		}

		public virtual void DataReceived( long nSenderSocketID,DSNetworkPacket oPacket )
		{
			const string sRoutineName = "DarkStrideToolbox.Network.DataReceived";
			DSNetworkPlayer oNetPlayer = null;

			try
			{
				if( m_oUs == null || nSenderSocketID != m_oUs.SocketID )
				{
					oNetPlayer = GetPlayer( nSenderSocketID );

					if( m_odDelegate_DataReceived != null )
					{
						m_odDelegate_DataReceived( oNetPlayer,oPacket );
					}
				}
			}
			catch( System.Exception oEx )
			{
				if( m_odDelegate_Error == null )
				{
					DSMisc.ShowErrors( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
				else
				{
					m_odDelegate_Error( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
			}
		}
		public virtual void DataReceived( DSNetworkPlayer oSendingPlayer,DSNetworkPacket oPacket )
		{
			if( m_odDelegate_DataReceived != null )
			{
				m_odDelegate_DataReceived( oSendingPlayer,oPacket );
			}
		}

		public virtual void SessionTerminated(object sender, SessionTerminatedEventArgs e)
		{
			const string sRoutineName = "DarkStrideToolbox.Network.SessionTerminated";

			try
			{
				// Well, this session is being terminated, let the user know
				/*if (e.Message.ResultCode ==  Microsoft.DirectX.DirectPlay.ResultCode.HostTerminatedSession)
				{
					MessageBox.Show("The Host has terminated this session.  This sample will now exit.", "Exiting", MessageBoxButtons.OK, MessageBoxIcon.Information);
				}
				else
				{
					MessageBox.Show("The session has been lost.  This sample will now exit.", "Exiting", MessageBoxButtons.OK, MessageBoxIcon.Information);
				}*/

				if( m_odDelegate_SessionTerminated != null )
				{
					m_odDelegate_SessionTerminated( e.Message.ResultCode );
				}
			}
			catch( System.Exception oEx )
			{
				if( m_odDelegate_Error == null )
				{
					DSMisc.ShowErrors( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
				else
				{
					m_odDelegate_Error( new System.Exception( sRoutineName + " Failed.",oEx ) );
				}
			}
		}
		
		public virtual void Connection( Microsoft.DirectX.DirectPlay.ResultCode oCode,string sDescription,bool bConnectionSuccessful )
		{
			if( m_odDelegate_Connection != null )
			{
				m_odDelegate_Connection( oCode,sDescription,bConnectionSuccessful );
			}
		}
		public virtual void GameListUpdate( FindHostsResponseInformation oGameInfo )
		{
			if( m_odDelegate_GameListUpdate != null )
			{
				m_odDelegate_GameListUpdate( oGameInfo );
			}
		}
		public virtual void CanceledGameSearch()
		{
			if( m_odDelegate_CanceledGameSearch != null )
			{
				m_odDelegate_CanceledGameSearch();
			}
		}


		public void RegisterDelegate_PlayerJoined( CallbackPlayerJoined odPlayerJoined )
		{
			const string sRoutineName = "DarkStrideToolbox.GameEngine.RegisterDelegate_PlayerJoined";

			try
			{
				m_odDelegate_PlayerJoined += odPlayerJoined;
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void RegisterDelegate_PlayerQuit( CallbackPlayerQuit odPlayerQuit )
		{
			const string sRoutineName = "DarkStrideToolbox.GameEngine.RegisterDelegate_PlayerQuit";

			try
			{
				m_odDelegate_PlayerQuit += odPlayerQuit;
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void RegisterDelegate_HostMigrated( CallbackHostMigrated odHostMigrated )
		{
			const string sRoutineName = "DarkStrideToolbox.GameEngine.RegisterDelegate_HostMigrated";

			try
			{
				m_odDelegate_HostMigrated += odHostMigrated;
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void RegisterDelegate_DataReceived( CallbackDataReceived odDataReceived )
		{
			const string sRoutineName = "DarkStrideToolbox.GameEngine.RegisterDelegate_DataReceived";

			try
			{
				m_odDelegate_DataReceived += odDataReceived;
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void RegisterDelegate_SessionTerminated( CallbackSessionTerminated odSessionTerminated )
		{
			const string sRoutineName = "DarkStrideToolbox.GameEngine.RegisterDelegate_SessionTerminated";

			try
			{
				m_odDelegate_SessionTerminated += odSessionTerminated;
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}

		public void RegisterDelegate_Connection( CallbackConnection odConnection )
		{
			m_odDelegate_Connection += odConnection;
		}

		public void RegisterDelegate_Error( CallbackError odError )
		{
			const string sRoutineName = "DarkStrideToolbox.Input.RegisterDelegate_Error";

			try
			{
				m_odDelegate_Error += odError;
			}
			catch( System.Exception oEx )
			{
				throw new System.Exception( sRoutineName + " Failed.",oEx );
			}
		}
		public void RegisterDelegate_GameListUpdate( CallbackGameListUpdate odGameListUpdate )
		{
			m_odDelegate_GameListUpdate += odGameListUpdate;
		}
		public void RegisterDelegate_CanceledGameSearch( CallbackCanceledGameSearch odCanceledGameSearch )
		{
			m_odDelegate_CanceledGameSearch += odCanceledGameSearch;
		}
		#endregion


		#region Properties
		public int DefaultPort
		{
			get
			{
				return( m_nDefaultPort );
			}
			set
			{
				m_nDefaultPort = value;
			}
		}
		public bool IsInitialized
		{
			get
			{
				return( m_oPeerObject != null );
			}
		}
		public DSSortedList Players
		{
			get
			{
				return( m_oPlayerList );
			}
		}
		public virtual bool IAmTheHost
		{
			get
			{
				return( m_oUs.IAmHost == true );
			}
		}
		public long LocalSocketID
		{
			get
			{
				return( m_oUs.SocketID );
			}
		}
		public DSNetworkPlayer Me
		{
			get
			{
				return( m_oUs );
			}
		}
		public Address Address
		{
			get
			{
				return( m_oOurAddress );
			}
		}
		public bool SearchingForGames
		{
			get
			{
				return( m_bSearchingForGames );
			}
		}
		public double SecondsBetweenPings
		{
			get
			{
				return( m_nSecondsBetweenPings );
			}
			set
			{
				m_nSecondsBetweenPings = value;
			}
		}
		public DSNetworkPlayer PlayerWhoIsHost
		{
			get
			{

				return( m_oPlayerWhoIsHost );
			}
		}
		public bool Connected
		{
			get
			{
				return( m_bConnected );
			}
		}

		//07/21/2007 Chris Hill  Added a debug mode for tracking.
        public bool DebugMode
        {
            set
            {
                //Are we turning it on when it was off?
                if (m_oDebug.DebugMode == false && value == true)
                {
                    //Then write a header
                    m_oDebug.WriteToDebug( value, 0,
                        "\r\n\r\nStarting Debug Log File For DSNetworkWrapper\r\n" +
                        "--------------------------------------------");
                }

                m_oDebug.DebugMode = value;
            }
            get
            {
                return (m_oDebug.DebugMode);
            }
        }
        public string DebugPath
        {
            get
            {
                return (m_oDebug.DebugPath);
            }
            set
            {
                m_oDebug.DebugPath = value;

                //We turn off then on the debug mode so that it writes a new header to the log
                if (m_oDebug.DebugMode == true)
                {
                    this.DebugMode = false;
                    this.DebugMode = true;
                }
            }
        }

        //Network statistics
        public double Latency
        {
            get
            {
                DSNetworkPlayer oLoopPlayer = null;
                DSSortedList oEntityClone = null;
                double nMinPing = -1;


                lock (m_oPlayerList)
                {
                    oEntityClone = m_oPlayerList.Clone();
                }
                for (int nPlrIndex = 0; nPlrIndex < oEntityClone.Count; nPlrIndex++)
                {
                    oLoopPlayer = (DSNetworkPlayer)m_oPlayerList.GetByIndex(nPlrIndex);

                    if (oLoopPlayer.IsMe == false && oLoopPlayer.LastPingInSeconds > 0 )
                    {
                        if (nMinPing == -1)
                        {
                            nMinPing = oLoopPlayer.LastPingInSeconds;
                        }
                        else
                        {
                            nMinPing = DSMisc.Min( oLoopPlayer.LastPingInSeconds,nMinPing );
                        }
                    }
                }


                return (nMinPing);
            }
        }
		#endregion
    }

    #region Helper Classes
    public class DSNetworkPlayer
	{
		#region Properties
		private string m_sName = "";
		private long m_nSocketID = -1;
		private bool m_bIAmHost = false;
		private bool m_bIsMe = false;
		private double m_nLastPingInSeconds = 0;
		private long m_nLastPingTime = 0;
        private long m_nJoinTime = 0;
        private long m_nLastMsgTime = 0;
		private long m_nPingStartCounter = 0;
		private long m_nPingStopCounter = 0;
		private EndPoint m_oUDPEndPoint = null;

        private long m_nHighestMessageIDReceived = 0;
        private long m_nNumberOfMessagesRecieved = 0;

        private DSNetworkObsoletePacketTracker m_oObsoletePacketTracker = new DSNetworkObsoletePacketTracker();
		#endregion

		public DSNetworkPlayer(){}
        
		#region Properties
		public string Name
		{
			get
			{
				return( m_sName );
			}
			set
			{
				m_sName = value;
			}
		}
		public long SocketID
		{
			get
			{
				return( m_nSocketID );
			}
			set
			{
				m_nSocketID = value;
			}
		}
		public bool IAmHost
		{
			get
			{
				return( m_bIAmHost );
			}
			set
			{
				m_bIAmHost = value;
			}
		}
		public bool IsMe
		{
			get
			{
				return( m_bIsMe );
			}
			set
			{
				m_bIsMe = value;
			}
		}
		public double LastPingInSeconds
		{
			get
			{
				return( m_nLastPingInSeconds );
			}
			set
			{
				m_nLastPingInSeconds = value;
			}
		}
        public long LastPingTime
		{
			get
			{
				return( m_nLastPingTime );
			}
			set
			{
				m_nLastPingTime = value;
			}
		}
        public long JoinTime
		{
			get
			{
				return( m_nJoinTime );
			}
			set
			{
				m_nJoinTime = value;
			}
		}
        public long LastMsgTime
        {
            get
            {
                return (m_nLastMsgTime);
            }
            set
            {
                m_nLastMsgTime = value;
            }
        }
		public long PingStartCounter
		{
			get
			{
				return( m_nPingStartCounter );
			}
			set
			{
				m_nPingStartCounter = value;
			}
		}
		public long PingStopCounter
		{
			get
			{
				return( m_nPingStopCounter );
			}
			set
			{
				m_nPingStopCounter = value;
			}
		}
		public EndPoint UDPEndPoint
		{
			get
			{
				return( m_oUDPEndPoint );
			}
			set
			{
				m_oUDPEndPoint = value;
			}
		}

        public long HighestMessageIDReceived
        {
            get
            {
                return (m_nHighestMessageIDReceived);
            }
            set
            {
                m_nHighestMessageIDReceived = value;
            }
        }
        public long NumberOfMessagesRecieved
        {
            get
            {
                return (m_nNumberOfMessagesRecieved);
            }
            set
            {
                m_nNumberOfMessagesRecieved = value;
            }
        }

        public DSNetworkObsoletePacketTracker ObsoletePacketTracker
        {
            get
            {
                return (m_oObsoletePacketTracker);
            }
            set
            {
                m_oObsoletePacketTracker = value;
            }
        }
		#endregion
	}
	internal class DSNetworkGuarenteedPacketACKsSent
	{
		#region Properties
		private long m_nMessageID = 0;
		private double m_nTimeLastSent = 0;
		#endregion


		#region Properties
		public long MessageID
		{
			get
			{
				return( m_nMessageID );
			}
			set
			{
				m_nMessageID = value;
			}
		}
		public double TimeLastSent
		{
			get
			{
				return( m_nTimeLastSent );
			}
			set
			{
				m_nTimeLastSent = value;
			}
		}
		#endregion
	}
	internal class DSNetworkGuarenteedPacket
	{
		#region Properties
		private DSNetworkPacket m_oPacketSent = null;
		private double m_nTimeLastSent = 0;
		private double m_nTimeFirstACKSent = 0;
		#endregion


		#region Properties
		public DSNetworkPacket Packet
		{
			get
			{
				return( m_oPacketSent );
			}
			set
			{
				m_oPacketSent = value;
			}
		}
		public double TimeLastSent
		{
			get
			{
				return( m_nTimeLastSent );
			}
			set
			{
				m_nTimeLastSent = value;
			}
		}
		public double TimeFirstACKSent
		{
			get
			{
				return( m_nTimeFirstACKSent );
			}
			set
			{
				m_nTimeFirstACKSent = value;
			}
		}
		#endregion
	}
    //Packet obsoleteness is to take care of you receiving old location updates after you've gotten new ones.
	internal class DSNetworkExcludeOldPackets
	{
		#region Properties
		private long m_nPacketMessageType = -1;
		private double m_nTimeLastMessageReceived = 0;
		private long m_nMessageIDOfLastPacket = -1;
		#endregion


		#region Properties
		public long PacketMessageType
		{
			get
			{
				return( m_nPacketMessageType );
			}
			set
			{
				m_nPacketMessageType = value;
			}
		}
		public double TimeLastMessageReceived
		{
			get
			{
				return( m_nTimeLastMessageReceived );
			}
			set
			{
				m_nTimeLastMessageReceived = value;
			}
		}
		public long MessageIDOfLastPacket
		{
			get
			{
				return( m_nMessageIDOfLastPacket );
			}
			set
			{
				m_nMessageIDOfLastPacket = value;
			}
		}
		#endregion
    }
    //Packet deduping is removing dupes of a packet so you only get the message once
    internal class DSNetworkPacketHistorys
    {
        #region Properties
        private System.Collections.Generic.SortedList<long, System.Collections.Generic.SortedList<long, DSNetworkPacketHistory>> m_oMessageHistorys = null;
        private DateTime m_dtLastCleanOut = DateTime.MinValue;
        #endregion


        public DSNetworkPacketHistorys()
        {
            m_oMessageHistorys = new System.Collections.Generic.SortedList<long, System.Collections.Generic.SortedList<long,DSNetworkPacketHistory>>();
        }
        //Returns weather or not the message is a dupe
        public bool IsMessageADupe( long nMessageType, long nMessageID )
        {
            System.Collections.Generic.SortedList<long, DSNetworkPacketHistory> oTypeHistory = null;
            DSNetworkPacketHistory oMsgHistory = null;
            bool bMessageIsADupe = false;


            lock (m_oMessageHistorys)
            {
                //Find the collection to store it in
                if (m_oMessageHistorys.ContainsKey(nMessageType) == true)
                {
                    oTypeHistory = m_oMessageHistorys[nMessageType];
                }
                else
                {
                    oTypeHistory = new System.Collections.Generic.SortedList<long,DSNetworkPacketHistory>();
                    m_oMessageHistorys.Add(nMessageType, oTypeHistory);
                }
            }

            lock (oTypeHistory)
            {
                //Ok, we've got our collection so store it
                if (oTypeHistory.ContainsKey(nMessageID) == true)
                {
                    bMessageIsADupe = true;
                }
                else
                {
                    oMsgHistory = new DSNetworkPacketHistory();
                    oMsgHistory.MessageID = nMessageID;
                    oMsgHistory.MessageArrived = DateTime.Now;
                    oTypeHistory.Add(nMessageID,oMsgHistory);
                }
            }


            return (bMessageIsADupe);
        }
        public void CleanOutOldMessages()
        {
            DSNetworkPacketHistory oLoopMsgHistory = null;
            DateTime oNow = DateTime.Now;
            TimeSpan oSpan = TimeSpan.MinValue;
            System.Collections.Generic.List<System.Collections.Generic.SortedList<long, DSNetworkPacketHistory>> oTempHistory = new System.Collections.Generic.List<System.Collections.Generic.SortedList<long, DSNetworkPacketHistory>>();

            
            //Don't clean out to often
            oSpan = oNow - m_dtLastCleanOut;
            if (oSpan.TotalSeconds > 15.0)
            {
                m_dtLastCleanOut = DateTime.Now;

                //Get a list of histories we know won't be changing
                lock (m_oMessageHistorys)
                {
                    foreach (System.Collections.Generic.SortedList<long,DSNetworkPacketHistory> oLoopTypeHistory in m_oMessageHistorys.Values)
                    {
                        oTempHistory.Add(oLoopTypeHistory);
                    }
                }

                //Now we know we have a colleection of histories that won't be changing
                foreach (System.Collections.Generic.SortedList<long, DSNetworkPacketHistory> oLoopTypeHistory in oTempHistory)
                {
                    //Walk through every message history in this type list
                    lock (oLoopTypeHistory)
                    {
                        for (int nMsgIdx = 0; nMsgIdx < oLoopTypeHistory.Count; nMsgIdx++)
                        {
                            if (nMsgIdx >= oLoopTypeHistory.Count) { break; }
                            oLoopMsgHistory = oLoopTypeHistory.Values[nMsgIdx];

                            oSpan = oNow - oLoopMsgHistory.MessageArrived;
                            if (oSpan.TotalSeconds > 15.0)
                            {
                                oLoopTypeHistory.RemoveAt(nMsgIdx);
                            }
                        }
                    }
                }
            }
        }


        #region Properties
        #endregion
    }
    internal class DSNetworkPacketHistory
    {
        #region Properties
        private long m_nMessageID = -1;
        private DateTime m_dtMessageArrived = DateTime.MinValue;
        #endregion


        #region Properties
        public long MessageID
        {
            get
            {
                return (m_nMessageID);
            }
            set
            {
                m_nMessageID = value;
            }
        }
        public DateTime MessageArrived
        {
            get
            {
                return (m_dtMessageArrived);
            }
            set
            {
                m_dtMessageArrived = value;
            }
        }
        #endregion
    }
    public class DSNetworkObsoletePacketTracker
    {
        #region Properties
        private System.Collections.Generic.SortedList<long, long> m_oOldPacketsReceived = new System.Collections.Generic.SortedList<long, long>();
        #endregion

        public bool IsPacketObsolete(long nMsgType, long nMessageID)
        {
            long nTempMsgID = 0;
            bool bIsObsolete = false;


            if (m_oOldPacketsReceived.ContainsKey(nMsgType) == true)
            {
                nTempMsgID = m_oOldPacketsReceived[nMsgType];

                if (nMessageID <= nTempMsgID)
                {
                    bIsObsolete = true;
                }
            }


            return (bIsObsolete);
        }
        public void RecordPacket(long nMsgType, long nMessageID)
        {
            if (m_oOldPacketsReceived.ContainsKey(nMsgType) == true)
            {
                m_oOldPacketsReceived.Remove(nMsgType);
            }

            m_oOldPacketsReceived.Add(nMsgType, nMessageID);
        }
        
        #region Properties
        #endregion
    }
    #endregion
}